# Copyright (C) 2009, 2010 Canonical Ltd
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import datetime
import os
import unittest

from apache_openid.handlers.openid.mixins import ProvidersMixin
from apache_openid.handlers.openidteams.authenticator import TeamsAuthenticator
from apache_openid.handlers.openidteams.mixins import CookiedTeamsMixin
from apache_openid.utils.mock import Options, Session

__all__ = ['CookiedTeamsMixinTest', 'ProvidersMixinTest',
    'TeamsAuthenticatorMixinTest']


class CookiedTeamsMixinTest(unittest.TestCase):

    def setUp(self):
        self.mixin = CookiedTeamsMixin()
        overrides = {
            'authorized-teams': "canonical-isd-hackers bzr-core",
        }
        self.mixin.options = Options(overrides)

    def test_cache_lifetime(self):
        self.assertEqual(
            self.mixin.cache_lifetime,
            datetime.timedelta(seconds=43200))
        self.mixin.options = Options({'team-cache-lifetime': 'foobar'})
        self.assertEqual(
            self.mixin.cache_lifetime,
            datetime.timedelta(seconds=43200))

    def test_authorized_teams_from_options(self):
        self.assertEqual(
            self.mixin.get_authorized_teams_from_options(),
            ['canonical-isd-hackers', 'bzr-core'])

    def test_expiring_team_cache(self):
        self.mixin.session = Session()
        self.mixin.cookied_teams = ['ubuntumembers', 'motu']
        sorted_teams = ['ubuntumembers', 'motu']
        sorted_teams.sort()
        self.assertEqual(
            self.mixin.cookied_teams,
            sorted_teams)
        # Set expiry times back to the past
        self.mixin.session['cookied_teams']['motu'] = datetime.datetime.now() - datetime.timedelta(minutes=1)
        self.mixin.session['cookied_teams']['ubuntumembers'] = datetime.datetime.now() - datetime.timedelta(minutes=1)
        self.assertEqual(
            self.mixin.cookied_teams,
            [])


class ProvidersMixinTest(unittest.TestCase):

    def setUp(self):
        self.mixin = ProvidersMixin()
        overrides = {
            'allowed-op-list-url': "file://%s" % os.path.abspath(os.path.join(os.path.dirname(__file__), 'allowed-ops.txt')),
        }
        self.mixin.options = Options(overrides)

    def test_allowed_ops(self):
        self.assertEqual(
            self.mixin.allowed_ops,
            {'Launchpad': 'login.launchpad.net'})


class TeamsAuthenticatorMixinTest(unittest.TestCase):
    
    def setUp(self):
        overrides = {
            'authorized-teams': "canonical-isd-hackers bzr-core",
            'authorized-teams-list-url': "file://%s" % os.path.abspath(os.path.join(os.path.dirname(__file__), 'authorized-teams.txt')),
        }
        options = Options(overrides)
        self.authenticator = TeamsAuthenticator(None, options, None)

    def test_authorized_teams_from_url(self):
        self.assertEqual(
            self.authenticator.get_authorized_teams_from_url(
                self.authenticator.options.get('authorized-teams-list-url')),
            ['ubuntumembers', 'ubuntu-core-dev', 'motu'])

    def test_authorized_teams_from_url_error(self):
        url = 'http://www.non-existing-host.com/this/page/doesnt/exist/'
        self.assertEqual(
            self.authenticator.get_authorized_teams_from_url(url),
            [])

    def test_authorized_teams(self):
        self.assertEqual(
            self.authenticator.authorized_teams,
            ['canonical-isd-hackers', 'bzr-core', 'ubuntumembers', 'ubuntu-core-dev', 'motu'])
