/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * raseq - count missing sequence numbers
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <argus_filter.h>
#include <raseq.h>


int RaInitialized = 0;

void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      RaInitialized++;
      if (!(RaSeqQueue = RaNewQueue()))
         ArgusLog (LOG_ERR, "ArgusClientInit %s\n", strerror(errno));
   }
}

int RaParseCompleting = 0;
unsigned int ArgusSequenceNumber = 0;
unsigned int ArgusMissingSeqNum  = 0;
int ArgusSeqError = 0;
int RaTotalOutofOrder = 0;
int RaTotalRecords = 0;

int RaSendArgusRecord(struct ArgusRecordStore *argus) {return 0;}


void
RaParseComplete (int sig)
{
   struct RaQueueHeader *obj = NULL, *last = NULL;
   
   if ((sig >= 0) && (!RaParseCompleting)) {
      RaParseCompleting++;
      if ((RaGetQueueCount(RaSeqQueue)) > 1) {
         while ((obj = (struct RaQueueHeader *) RaPopQueue(RaSeqQueue))) {
            if (last != NULL) {
               ArgusMissingSeqNum += obj->start - last->end;
               ArgusFree(last);
               last = NULL;
            }

            last = obj;
         }

         if (obj != NULL)
            ArgusFree(obj);
      }

      if (ArgusMissingSeqNum > 0)
         printf ("%d missing sequence numbers\n", ArgusMissingSeqNum);

      if (aflag) {
         printf ("RaTotalRecords    %d\n", RaTotalRecords);
         printf ("RaTotalOutofOrder %d\n", RaTotalOutofOrder);
      }

      fflush (stdout);
   }

   exit (ArgusMissingSeqNum);
}

void
ArgusClientTimeout ()
{
}

void
parse_arg (int argc, char**argv)
{ 
}


void
usage ()
{
   extern char version[];
   fprintf (stderr, "Ratemplate Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusProgramName);

   fprintf (stderr, "options: -b                dump packet-matching code.\n");
   fprintf (stderr, "         -D <level>        specify debug level\n");
   fprintf (stderr, "         -F <conffile>     read configuration from <conffile>.\n");
   fprintf (stderr, "         -h                print help.\n");
   fprintf (stderr, "         -r <file>         read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -S <host[:port]>  specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "         -t <timerange>    specify <timerange> for reading records.\n");
   fprintf (stderr, "                  format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                           timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                               mm/dd[/yy]\n");
   fprintf (stderr, "                                               -%%d{yMhdms}\n");
   fprintf (stderr, "         -T <secs>         attach to remote server for T seconds.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth>    specify <user/auth> authentication information.\n");
#endif
   exit(1);
}


struct RaQueueHeader *RaLastObj = NULL;
struct RaQueueHeader *RaFindQueueEntry (int);



void
RaProcessRecord (struct ArgusRecord *argus)
{
   struct RaQueueHeader *obj = NULL;
   struct ArgusRecordHeader *ahdr = &argus->ahdr;
   unsigned int thisSeq = ahdr->seqNumber;
   int thisSeqOutofOrder = 0;

   if (!(argus->ahdr.type & ARGUS_MAR)) {
      if ((ArgusSequenceNumber + 1) != thisSeq) {
         if (ArgusSequenceNumber != 0)
            thisSeqOutofOrder++;

         if ((RaLastObj = RaFindQueueEntry(thisSeq)) != NULL) {
            RaLastObj->end = thisSeq;
            if (RaLastObj->nxt && (RaLastObj->nxt->start == (thisSeq + 1))) {
               RaLastObj->end = RaLastObj->nxt->end;
               RaRemoveFromQueue (RaSeqQueue, (struct ArgusQueueHeader *) RaLastObj->nxt);
            }
            
         } else {
            if ((obj = (struct RaQueueHeader *) ArgusCalloc (1, sizeof(*obj))) != NULL) {
               obj->start = thisSeq;
               obj->end   = thisSeq;
               RaAddToQueue(RaSeqQueue, (struct ArgusQueueHeader *) obj);
               if ((obj->end + 1) == (obj->nxt->start)) {
                  obj->end = obj->nxt->end;
                  RaRemoveFromQueue (RaSeqQueue, (struct ArgusQueueHeader *) obj->nxt);
               }
               
               RaLastObj = obj;
            }
         }

      } else {
         if (RaLastObj != NULL) {
            RaLastObj->end = thisSeq;
            if (RaLastObj->nxt->start == (thisSeq + 1)) {
               RaLastObj->end = RaLastObj->nxt->end;
               RaRemoveFromQueue(RaSeqQueue, (struct ArgusQueueHeader *) RaLastObj->nxt);
            }

         } else {
            struct RaQueueHeader *obj = NULL;

            if ((obj = (struct RaQueueHeader *) ArgusCalloc (1, sizeof(*obj))) != NULL) {
               obj->start = thisSeq;
               obj->end   = thisSeq;
               RaAddToQueue(RaSeqQueue, (struct ArgusQueueHeader *) obj);
               if ((obj->end + 1) == (obj->nxt->start)) {
                  obj->end = obj->nxt->end;
                  RaRemoveFromQueue (RaSeqQueue, (struct ArgusQueueHeader *) obj->nxt);
               }
               
               RaLastObj = obj;
            }
         }
      }

      ArgusSequenceNumber = ahdr->seqNumber;

      if (aflag) {
         RaTotalRecords++;
         if (thisSeqOutofOrder)
            RaTotalOutofOrder++;
      }
   }
}


struct RaQueueHeader *
RaFindQueueEntry (int seq)
{
   struct RaQueueHeader *retn = NULL;
   int found = 0;

   if (RaSeqQueue && RaSeqQueue->count) {
      if ((retn = (struct RaQueueHeader *) RaSeqQueue->start) != NULL) {
         do {
            if ((retn->end + 1) == seq) {
               found++;
               break;
            }

            retn = retn->nxt;
         } while (!found && (retn != (struct RaQueueHeader *) RaSeqQueue->start));

         if (!found)
            retn = NULL;
      }
   }

   return (retn);
}
