/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

*/

#include <cdi.h>

#include "dmemory.h"
#include "process_int.h"
#include "interpol.h"
#include "datetime.h"
#include "printinfo.h"

static int
readnextpos(FILE *fp, int calendar, JulianDate *juldate, double *xpos, double *ypos)
{
  *xpos = 0;
  *ypos = 0;

  int year = 0, month = 0, day = 0, hour = 0, minute = 0, second = 0;
  int stat = fscanf(fp, "%d-%d-%d %d:%d:%d %lf %lf", &year, &month, &day, &hour, &minute, &second, xpos, ypos);

  if (stat != EOF)
    {
      auto date = cdiEncodeDate(year, month, day);
      auto time = cdiEncodeTime(hour, minute, second);
      *juldate = julianDateEncode(calendar, date, time);
    }

  return stat;
}

void *
Intgridtraj(void *process)
{
  int varID, levelID;
  int64_t vdate;
  int vtime;
  size_t nmiss = 0;
  double xpos, ypos;
  int calendar = CALENDAR_STANDARD;

  cdoInitialize(process);

  operatorInputArg("filename with grid trajectories");
  operatorCheckArgc(1);

  auto posfile = cdoOperatorArgv(0).c_str();
  auto fp = fopen(posfile, "r");
  if (fp == nullptr) cdoAbort("Open failed on %s!", posfile);

  JulianDate juldate;
  readnextpos(fp, calendar, &juldate, &xpos, &ypos);

  const auto streamID1 = cdoOpenRead(0);
  const auto vlistID1 = cdoStreamInqVlist(streamID1);

  const auto nvars = vlistNvars(vlistID1);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);
  Field field1, field2;
  field1.resize(gridsizemax);
  field2.resize(1);

  double **vardata1 = (double **) Malloc(nvars * sizeof(double *));
  double **vardata2 = (double **) Malloc(nvars * sizeof(double *));

  for (varID = 0; varID < nvars; varID++)
    {
      const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
      const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
      vardata1[varID] = (double *) Malloc(gridsize * nlevel * sizeof(double));
      vardata2[varID] = (double *) Malloc(gridsize * nlevel * sizeof(double));
    }

  const auto gridID2 = gridCreate(GRID_TRAJECTORY, 1);
  gridDefXsize(gridID2, 1);
  gridDefYsize(gridID2, 1);
  gridDefXvals(gridID2, &xpos);
  gridDefYvals(gridID2, &ypos);

  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto ngrids = vlistNgrids(vlistID1);
  for (int index = 0; index < ngrids; index++)
    {
      const auto gridID1 = vlistGrid(vlistID1, index);

      if (gridInqType(gridID1) != GRID_LONLAT && gridInqType(gridID1) != GRID_GAUSSIAN)
        cdoAbort("Unsupported grid type: %s", gridNamePtr(gridInqType(gridID1)));

      vlistChangeGridIndex(vlistID2, index, gridID2);
    }

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  CdoStreamID streamID2 = CDO_STREAM_UNDEF;

  int tsID = 0;
  auto nrecs = cdoStreamInqTimestep(streamID1, tsID++);
  auto juldate1 = julianDateEncode(calendar, taxisInqVdate(taxisID1), taxisInqVtime(taxisID1));
  for (int recID = 0; recID < nrecs; recID++)
    {
      cdoInqRecord(streamID1, &varID, &levelID);
      const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
      const auto offset = gridsize * levelID;
      double *single1 = vardata1[varID] + offset;
      cdoReadRecord(streamID1, single1, &nmiss);
      if (nmiss) cdoAbort("Missing values unsupported for this operator!");
    }

  int tsIDo = 0;
  while (julianDateToSeconds(juldate1) <= julianDateToSeconds(juldate))
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID++);
      if (nrecs == 0) break;
      auto juldate2 = julianDateEncode(calendar, taxisInqVdate(taxisID1), taxisInqVtime(taxisID1));

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          recList[recID].varID = varID;
          recList[recID].levelID = levelID;
          recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;

          const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
          const auto offset = gridsize * levelID;
          double *single2 = vardata2[varID] + offset;
          cdoReadRecord(streamID1, single2, &nmiss);
          if (nmiss) cdoAbort("Missing values unsupported for this operator!");
        }

      while (julianDateToSeconds(juldate) < julianDateToSeconds(juldate2))
        {
          if (julianDateToSeconds(juldate) >= julianDateToSeconds(juldate1)
              && julianDateToSeconds(juldate) < julianDateToSeconds(juldate2))
            {
              if (streamID2 == CDO_STREAM_UNDEF)
                {
                  streamID2 = cdoOpenWrite(1);
                  cdoDefVlist(streamID2, vlistID2);
                }

              julianDateDecode(calendar, juldate, vdate, vtime);
              taxisDefVdate(taxisID2, vdate);
              taxisDefVtime(taxisID2, vtime);
              cdoDefTimestep(streamID2, tsIDo++);

              const auto fac1
                  = julianDateToSeconds(julianDateSub(juldate2, juldate)) / julianDateToSeconds(julianDateSub(juldate2, juldate1));
              const auto fac2
                  = julianDateToSeconds(julianDateSub(juldate, juldate1)) / julianDateToSeconds(julianDateSub(juldate2, juldate1));
              /*
              printf("      %f %f %f %f %f\n", julianDateToSeconds(juldate),
                                               julianDateToSeconds(juldate1),
                                               julianDateToSeconds(juldate2),
              fac1, fac2);
              */
              for (int recID = 0; recID < nrecs; recID++)
                {
                  varID = recList[recID].varID;
                  levelID = recList[recID].levelID;
                  const auto missval = vlistInqVarMissval(vlistID1, varID);
                  const auto gridID1 = vlistInqVarGrid(vlistID1, varID);
                  const auto gridsize = gridInqSize(gridID1);
                  const auto offset = gridsize * levelID;
                  double *single1 = vardata1[varID] + offset;
                  double *single2 = vardata2[varID] + offset;

                  for (size_t i = 0; i < gridsize; i++) field1.vec[i] = single1[i] * fac1 + single2[i] * fac2;

                  field1.grid = gridID1;
                  field1.nmiss = nmiss;
                  field1.missval = missval;
                  field2.grid = gridID2;
                  field2.nmiss = 0;

                  intgridbil(field1, field2);

                  cdoDefRecord(streamID2, varID, levelID);
                  cdoWriteRecord(streamID2, field2.vec.data(), field2.nmiss);
                }
            }
          if (readnextpos(fp, calendar, &juldate, &xpos, &ypos) == EOF) break;
          gridDefXvals(gridID2, &xpos);
          gridDefYvals(gridID2, &ypos);
        }

      juldate1 = juldate2;
      for (varID = 0; varID < nvars; varID++)
        {
          double *vardatap = vardata1[varID];
          vardata1[varID] = vardata2[varID];
          vardata2[varID] = vardatap;
        }
    }

  if (tsIDo == 0)
    {
      julianDateDecode(calendar, juldate, vdate, vtime);
      cdoWarning("Date/time %s %s not found!", dateToString(vdate).c_str(), timeToString(vtime).c_str());
    }

  fclose(fp);
  if (streamID2 != CDO_STREAM_UNDEF) cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  for (varID = 0; varID < nvars; varID++)
    {
      Free(vardata1[varID]);
      Free(vardata2[varID]);
    }
  Free(vardata1);
  Free(vardata2);

  cdoFinish();

  return nullptr;
}
