/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define _BSD_SOURCE /* strdup() */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "cdw_fs.h"
#include "config.h"
#include "cdw_main_window.h"
#include "cdw_config.h"
#include "gettext.h"
#include "cdw_file_manager.h"
#include "cdw_debug.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_window.h"
#include "cdw_text_file_viewer.h"
#include "cdw_drive.h"
#include "cdw_cdio_drives.h"
#include "cdw_utils.h"

extern cdw_config_t global_config; /* main cdw configuration variable */

static void cdw_main_ui_init_values(void);
static void cdw_main_ui_recalculate_sizes(void);

/** \brief Number of items in main cdw menu on the left; does not include last NULL item */
#define MENU_ITEMS 10
/* in theory this is just a length of labels in menu, and indirectly
   width of menu; but it also (indirectly) influences width of
   "disc info" box below menu */
#define MENU_LABEL_LEN 24

/* \brief Main menu on the left side of main cdw window */
struct cdw_menu {

	MENU *menu;
	ITEM **items;

	/** \brief Subwindow for a view - the one that will get a border, and
	    the one that will be used in set_menu_win() */
	WINDOW *subwindow;

	/** \brief Subwindow for a menu, the one used in set_menu_sub() */
	WINDOW *menu_subwindow;

	int n_rows;
	int n_cols;
	int begin_y;
	int begin_x;
};


struct cdw_ui_view {
	WINDOW *subwindow;
	int n_rows;
	int n_cols;
	int begin_y;
	int begin_x;
};


static struct {
	struct cdw_menu main_menu;
	struct cdw_ui_view disc_info_view;
	struct cdw_ui_view files_list_view;
	struct cdw_ui_view volume_info_view;
	struct cdw_ui_view tooltips_view;

	WINDOW *window;
	int n_rows;
	int n_cols;
} cdw_ui;



static cdw_rv_t cdw_main_ui_main_menu_view_create(void);
static cdw_rv_t cdw_main_ui_files_list_view_create(void);
static cdw_rv_t cdw_main_ui_volume_info_view_create(void);
static cdw_rv_t cdw_main_ui_disc_info_view_create(void);
static cdw_rv_t cdw_main_ui_tooltips_view_create(void);

static void cdw_main_ui_volume_info_view_display_data_new(bool show_disc_info, size_t n_files, double used_space, double selected_size, int total_space);




/**
   \brief Initialize resources of main cdw window

   Initialize all windows and other data structures related to user interface
   that can be seen right after the application starts.

   \return CDW_MEM_ERROR if some malloc-related error occurred
   \return CDW_GEN_ERROR if some other error occurred
   \return CDW_OK if user interface was initialized properly
*/
cdw_rv_t cdw_main_ui_init(void)
{
	cdw_main_ui_init_values();

	/* main NCURSES window */
	cdw_ui.window = newwin(cdw_ui.n_rows, cdw_ui.n_cols, 0, 0);
	if (!cdw_ui.window) {
		cdw_main_ui_clean();
		cdw_vdm ("ERROR: failed to create main app window\n");
		return CDW_ERROR;
	}

	keypad(cdw_ui.window, TRUE);
	wbkgd(cdw_ui.window, A_NORMAL | COLOR_PAIR(CDW_COLORS_MAIN));

	/* left-hand, main app menu */
	cdw_rv_t crv = cdw_main_ui_main_menu_view_create();
	if (crv != CDW_OK) {
		cdw_main_ui_clean();
		cdw_vdm ("ERROR: failed to create main app menu\n");
		return CDW_ERROR;
	}

	mvwaddch(cdw_ui.window, 0, 1, ACS_RTEE);
	wattrset(cdw_ui.window, A_BOLD | COLOR_PAIR(CDW_COLORS_TITLE));
	mvwprintw(cdw_ui.window, 0, 2, " ::: %s %s ::: ", PACKAGE, VERSION);
	wattrset(cdw_ui.window, A_NORMAL | COLOR_PAIR(CDW_COLORS_MAIN));
	waddch(cdw_ui.window, ACS_LTEE);

	/* window in which selected files are displayed */
	crv = cdw_main_ui_files_list_view_create();
	if (crv != CDW_OK) {
		cdw_main_ui_clean();
		cdw_vdm ("ERROR: failed to create selected files list view\n");
		return CDW_ERROR;
	}

	/* area displaying summarized information about selected files
	   and disc capacity/usage/estimated future usage */
	crv = cdw_main_ui_volume_info_view_create();
	if (crv != CDW_OK) {
		cdw_main_ui_clean();
		cdw_vdm ("ERROR: failed to create selected files info view\n");
		return CDW_ERROR;
	}

	cdw_main_window_volume_info_view_update(-1, -1, true, global_config.follow_symlinks);
	//selected_files_size_mb = 0.0;
	//cdw_ui_volume_info_view_display_data();

	/* area displaying information about current optical disc */
	crv = cdw_main_ui_disc_info_view_create();
	if (crv != CDW_OK) {
		cdw_main_ui_clean();
		cdw_vdm ("ERROR: failed to create disc info view\n");
		return CDW_ERROR;
	}

	crv = cdw_main_ui_tooltips_view_create();
	if (crv != CDW_OK) {
		cdw_main_ui_clean();
		cdw_vdm ("ERROR: failed to create help view\n");
		return CDW_ERROR;
	}

	wrefresh(cdw_ui.window);

	return CDW_OK;
}





/**
   \brief Deallocate all resources related to main app window

   This function may be called even when no resources are allocated yet:
   this functions and all functions called check if resource was allocated,
   and if so, then proceeds with cleaning resource.
*/
void cdw_main_ui_clean(void)
{
	if (cdw_ui.main_menu.menu) {
		unpost_menu(cdw_ui.main_menu.menu);
		cdw_ncurses_delete_menu_and_items(&(cdw_ui.main_menu.menu));
		cdw_assert (!cdw_ui.main_menu.menu, "failed to set cdw_ui.main_menu.menu to NULL\n")
	}

	/* IMPORTANT: first clean subwindows, then parent window */

	/* FIXME: there may be a bug somewhere, steps to reproduce:
	   1. in cdw_curses_init() change this line:
	   "if ((lines < 24) || (cols < 80))" to
	   "if ((lines < 25) || (cols < 80))"
	   2. run cdw in 80x25 terminal
	   3. right after starting cdw press 'q' key to quit
	   4. in dialog window select 'ok' and press enter
	   5. cdw crashes

	   debugger points to first delwin() below. Note that the
	   bug doesn't occur in 2 circumstances:
	   1. if in dialog from point 4 above you select 'cancel'
	   and then you press 'q' again and then select 'ok'
	   2. if you run cdw in 81x26 terminal */

	cdw_window_delete(&cdw_ui.files_list_view.subwindow);
	cdw_window_delete(&cdw_ui.main_menu.menu_subwindow);
	cdw_assert (cdw_ui.main_menu.menu_subwindow == (WINDOW *) NULL,
		    "ERROR: delete() function didn't set a window to NULL\n");
	cdw_window_delete(&cdw_ui.main_menu.subwindow);
	cdw_window_delete(&cdw_ui.volume_info_view.subwindow);
	cdw_window_delete(&cdw_ui.disc_info_view.subwindow);
	cdw_window_delete(&cdw_ui.tooltips_view.subwindow);
	cdw_window_delete(&cdw_ui.window);

	return;
}





/**
   \brief Calculate values of some layout constraints and parameters

   Calculate values of variables that decide about height, width and position
   of subwindows in main cdw UI.

   This function must be used before cdw_ui_main_init() is called. Should be
   also called when application receives information that terminal size has
   changed (TODO: catching signal is to be implemented).
*/
void cdw_main_ui_recalculate_sizes(void)
{
	/* There are only few known limitations on UI layout:
	    - tooltips area at the bottom of window must have exactly 1 line;
	    - disc info view needs to display 4 lines + 2 lines of
	      window borders;
	    - main menu area must fit labels of certain length;
	    - minimal supported terminal size is 80 cols x 24 lines;

	   Height of volume information area is the same as disc info
	   area: 4 + 2, so I don't consider this height as an additional
	   basic factor.

	   Width of disc info area is the same as width of main menu - this
	   seems to be sufficient.

	   Width of volume size area and selected files area aren't very
	   limiting factors.

	   We have also remember about number of items in main menu, which
	   is 9 items + 2 lines of window borders. However quick calculation
	   shows that this is not an important factor:
	   24 lines of terminal - 1 line of tooltips - (4 + 2) lines of
	   disc info gives 17 lines - this is how much lines we have for
	   main menu area (and for selected files area). */

	cdw_ui.n_rows = LINES;
	cdw_ui.n_cols = COLS;

	/* -2: don't let letters in tooltips area be displayed very
	   close to window borders */
	cdw_ui.tooltips_view.n_cols = cdw_ui.n_cols - 2;
	cdw_ui.tooltips_view.n_rows = 1;
	cdw_ui.tooltips_view.begin_y = cdw_ui.n_rows - cdw_ui.tooltips_view.n_rows;
	cdw_ui.tooltips_view.begin_x = 1;

	/* disc info area needs 4 lines of usable space
	   (Type?, Empty?, Writable?, Erasable?) + 2 lines for borders */
        int lower_n_rows = 4 + 2;
	int upper_n_rows = cdw_ui.n_rows - lower_n_rows - cdw_ui.tooltips_view.n_rows;
	int left_n_cols = MENU_LABEL_LEN;
	int right_n_cols = cdw_ui.n_cols - left_n_cols;

	int lower_begin_y = cdw_ui.n_rows - lower_n_rows - cdw_ui.tooltips_view.n_rows;
	int upper_begin_y = 0;
	int right_begin_x = left_n_cols;
	int left_begin_x = 0;

	cdw_ui.main_menu.n_cols = left_n_cols;
	cdw_ui.main_menu.n_rows = upper_n_rows;
	cdw_ui.main_menu.begin_y = upper_begin_y;
	cdw_ui.main_menu.begin_x = left_begin_x;

	cdw_ui.files_list_view.n_rows = upper_n_rows;
	cdw_ui.files_list_view.n_cols = right_n_cols;
	cdw_ui.files_list_view.begin_y = upper_begin_y;
	cdw_ui.files_list_view.begin_x = right_begin_x;

	cdw_ui.volume_info_view.n_rows = lower_n_rows;
	cdw_ui.volume_info_view.n_cols = right_n_cols;
	cdw_ui.volume_info_view.begin_y = lower_begin_y;
	cdw_ui.volume_info_view.begin_x = right_begin_x;

	cdw_ui.disc_info_view.n_cols = left_n_cols;
	cdw_ui.disc_info_view.n_rows = lower_n_rows;
	cdw_ui.disc_info_view.begin_y = lower_begin_y;
	cdw_ui.disc_info_view.begin_x = left_begin_x;

	return;
}




static const char *menu_labels[] = {
	/* 2TRANS: label of item main menu: Add files from file system to
	   list of files that will be written to CD or to image files*/
	gettext_noop(" Add files "),
	/* 2TRANS: label of item in main menu: Remove dir(s)/file(s) from
	   list of files that will be written to CD or to image files */
	gettext_noop(" Delete files "),
	/* 2TRANS: label of item in main menu: write
	   selected dirs/files to optical disc */
	gettext_noop(" Write files to disc "),
	/* 2TRANS: label of item in main menu: create ISO9660 image file
	   using selected files/directories */
	gettext_noop(" Create image "),
	/* 2TRANS: label of item in main menu: write (burn) ISO9660 image
	   file to optical disc */
	gettext_noop(" Write image to disc "),
	/* 2TRANS: label of item in main menu: read content of
	   your audio CD / data CD / data DVD and write it to hard drive */
	gettext_noop(" Read disc "),
	/* 2TRANS: label of item in main menu: erase content of optical disc */
	gettext_noop(" Erase disc "),
	/* 2TRANS: label of item in main menu: perform verification of data */
	gettext_noop(" Verify data "),
	/* 2TRANS: label of item in main menu: change configuration of cdw */
	gettext_noop(" Configuration "),
	/* 2TRANS: label of item in main menu: exit cdw */
	gettext_noop(" Quit ") };




/**
   \brief Create and initialize main app menu and its subwindow

   Create and initialize main CDW menu that is visible in main UI.
   Create its subwindow.

   \return CDW_OK if everything went ok
   \return CDW_MEM_ERROR if some malloc() error occurred
   \return CDW_ERROR if some other error
*/
cdw_rv_t cdw_main_ui_main_menu_view_create(void)
{
	cdw_assert (cdw_ui.window, "calling the function for null main window\n");

	cdw_ui.main_menu.subwindow = derwin(cdw_ui.window,
					    cdw_ui.main_menu.n_rows,
					    cdw_ui.main_menu.n_cols,
					    cdw_ui.main_menu.begin_y,
					    cdw_ui.main_menu.begin_x);
	if (!cdw_ui.main_menu.subwindow) {
		cdw_vdm ("ERROR: failed to create subwindow for main menu\n");
		return CDW_ERROR;
	}
	cdw_ui.main_menu.menu_subwindow = derwin(cdw_ui.main_menu.subwindow,
						 cdw_ui.main_menu.n_rows - 2,
						 cdw_ui.main_menu.n_cols - 2,
						 1, 1);
	if (!cdw_ui.main_menu.menu_subwindow) {
		cdw_vdm ("ERROR: failed to create menu subwindow for main menu\n");
		return CDW_ERROR;
	}

	werase(cdw_ui.main_menu.subwindow);
	werase(cdw_ui.main_menu.menu_subwindow);

	/* no title for this subarea - it would be overwritten with
	   app name + version */
	cdw_window_add_strings(cdw_ui.main_menu.subwindow, (char *) NULL, (char *) NULL);

	cdw_ui.main_menu.items = (ITEM **) calloc(MENU_ITEMS + 1, sizeof(ITEM *));
	if (!cdw_ui.main_menu.items) {
		cdw_vdm ("ERROR: failed to allocate memory for menu items\n");
		return CDW_ERROR;
	}

	for (int i = 0; i < MENU_ITEMS; i++) {
		/* label too long will be simply truncated by menu window */
		/* cdw_ui.main_menu.labels[i][MENU_LABEL_LEN] = '\0'; */
		cdw_ui.main_menu.items[i] = new_item(_(menu_labels[i]), "");
	}
	cdw_ui.main_menu.items[MENU_ITEMS] = (ITEM *) NULL;

	cdw_ui.main_menu.menu = new_menu((ITEM **) cdw_ui.main_menu.items);
	if (!cdw_ui.main_menu.menu) {
		int e = errno;
		cdw_vdm ("ERROR: failed to create menu with new_menu(), errno is \"%s\"\n",
			 cdw_ncurses_error_string(e));
		return CDW_ERROR;
	}

	int r = set_menu_win(cdw_ui.main_menu.menu, cdw_ui.main_menu.subwindow);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to set menu win with set_menu_win(), return value is \"%s\"\n",
			 cdw_ncurses_error_string(r));
		return CDW_ERROR;
	}
	r = set_menu_sub(cdw_ui.main_menu.menu, cdw_ui.main_menu.menu_subwindow);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to set menu subwin with set_menu_sub(), return value is \"%s\"\n",
			 cdw_ncurses_error_string(r));
		return CDW_ERROR;
	}
	set_menu_fore(cdw_ui.main_menu.menu, COLOR_PAIR(CDW_COLORS_MENU) | A_REVERSE);
	set_menu_back(cdw_ui.main_menu.menu, COLOR_PAIR(CDW_COLORS_MENU));
	r = set_menu_mark(cdw_ui.main_menu.menu, "");
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to set menu mark with set_menu_mark(), return value is \"%s\"\n",
			 cdw_ncurses_error_string(r));
		return CDW_ERROR;
	}

	r = post_menu(cdw_ui.main_menu.menu);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to post menu with post_menu(), return value is \"%s\"\n",
			 cdw_ncurses_error_string(r));
		return CDW_ERROR;
	}

	return CDW_OK;
}





/**
   \brief Create 'Selected files' part of main application window

   In case of errors function does not deallocate resources that it
   allocated (selected_files_sub).

   \return CDW_OK on success
   \return CDW_ERROR if some error occurred
*/
cdw_rv_t cdw_main_ui_files_list_view_create(void)
{
	cdw_assert (cdw_ui.window, "ERROR: calling the function for null main window\n");

	cdw_ui.files_list_view.subwindow = derwin(cdw_ui.window,
						  cdw_ui.files_list_view.n_rows,
						  cdw_ui.files_list_view.n_cols,
						  cdw_ui.files_list_view.begin_y,
						  cdw_ui.files_list_view.begin_x);

	if (!cdw_ui.files_list_view.subwindow) {
		cdw_vdm ("ERROR: failed to create files list view subwindow with derwin()\n");
		return CDW_ERROR;
	}

	keypad(cdw_ui.files_list_view.subwindow, TRUE);
	cdw_window_add_strings(cdw_ui.files_list_view.subwindow,
			       /* 2TRANS: this is title of biggest area in
				  cdw UI that shows files selected for
				  writing to cd or image file */
			       _("Selected files"), (char *) NULL);
	wattrset(cdw_ui.files_list_view.subwindow, A_NORMAL | COLOR_PAIR(CDW_COLORS_MAIN));

	cdw_rv_t crv = cdw_file_manager_create_selected_files_view(cdw_ui.files_list_view.subwindow);
	if (crv == CDW_OK) {
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: failed to create files list view with cdw_file_manager*() call()\n");
		/* file manager already displayed error message, caller will have
		   to decide what to do with allocated resources (subwindow)  */
		return CDW_ERROR;
	}
}





/**
   \brief Create subwindow with information about optical disc

   Create subwindow in which information about current disc is displayed.
   Display initial view in this subwindow.

   \return CDW_OK on success
   \return CDW_ERROR if some error occurred
*/
cdw_rv_t cdw_main_ui_disc_info_view_create(void)
{
	cdw_ui.disc_info_view.subwindow = derwin(cdw_ui.window,
						 cdw_ui.disc_info_view.n_rows,
						 cdw_ui.disc_info_view.n_cols,
						 cdw_ui.disc_info_view.begin_y,
						 cdw_ui.disc_info_view.begin_x);

	if (!cdw_ui.disc_info_view.subwindow) {
		cdw_vdm ("ERROR: failed to create subwindow with derwin() for disc info view\n");
		return CDW_ERROR;
	}

	cdw_window_add_strings(cdw_ui.disc_info_view.subwindow,
			       /* 2TRANS: this is title of area displaying basic
				  information about disc currently in drive */
			       _("Disc info"), (char *) NULL);

	cdw_disc_t *disc = cdw_disc_new();

	/* no call to cdw_disc_get() when UI is created;
	   if there is no disc in drive, the user would be nagged to
	   insert disc into drive - that nagging wouldn't be welcomed
	   by the user

	   cdw_disc_get(disc); */

	cdw_main_ui_disc_info_view_display_data(disc);
	cdw_disc_delete(&disc);

	return CDW_OK;
}





/**
   \brief Display in subwindow current information about optical disc

   Read from proper variable information about current disc and display
   this information in 'Disc info' subwindow in main CDW window.

   The function does not retrieve from optical disc nor recalculate any data,
   it merely uses data existing in \p disc.

   \param disc - variable from which to take values to display
*/
void cdw_main_ui_disc_info_view_display_data(const cdw_disc_t *disc)
{
	cdw_assert (cdw_ui.disc_info_view.subwindow, "called the function for NULL subwindow\n");
	cdw_assert (disc, "called the function with NULL argument\n");

	/* "parameter name" column and "parameter value" column */
	const int pcol = 2;
	const int vcol = 14;

	const int row_h = 1;  /* "header" row */
	const int row_e = 2;  /* "empty" row */
	const int row_w = 3;  /* "writable" row */
	const int row_e2 = 4; /* "erasable" row */

	/* erase; not using werase(cdw_ui.disc_info_view.subwindow)
	   because it would erase subwindow borders too */
	mvwhline(cdw_ui.disc_info_view.subwindow, row_h, 1, ' ', cdw_ui.main_menu.n_cols - 2);
	mvwhline(cdw_ui.disc_info_view.subwindow, row_e, 1, ' ', cdw_ui.main_menu.n_cols - 2);
	mvwhline(cdw_ui.disc_info_view.subwindow, row_w, 1, ' ', cdw_ui.main_menu.n_cols - 2);
	mvwhline(cdw_ui.disc_info_view.subwindow, row_e2, 1, ' ', cdw_ui.main_menu.n_cols - 2);

	wrefresh(cdw_ui.disc_info_view.subwindow);
	wattrset(cdw_ui.disc_info_view.subwindow, A_BOLD | COLOR_PAIR(CDW_COLORS_MAIN));


	/* header:
	   if no disc in drive then "No disc"
	   if disc is empty then "<disc type>"
	   if disc is not empty then "<disc type>/<file system type>" */
	if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_UNKNOWN) {
		/* 2TRANS: "No disc" means "currently there is no disc in
		   drive or no information about disc has been collected
		   yet"; keep short */
		mvwprintw(cdw_ui.disc_info_view.subwindow, row_h, pcol, _("No disc"));
		int n_drives = cdw_cdio_drives_get_n_drives();
		if (n_drives == 0) {
			/* 2TRANS: "No drive" means "no optical drive
			   detected by cdw", "?" means that cdw is not
			   100% sure about this; keep short */
			mvwprintw(cdw_ui.disc_info_view.subwindow, row_h + 1, pcol, _("No drive (?)"));
		}
		/* 2TRANS: this is a hint about a hot key; keep short;
		   "refresh" means - (re)read disc information and display
		   it in disc information area */
		mvwprintw(cdw_ui.disc_info_view.subwindow, row_h + 2, pcol, _("'R' key - refresh"));
		return;
	} else {
		if (disc->state_empty == CDW_TRUE) {
			mvwprintw(cdw_ui.disc_info_view.subwindow, row_h, pcol, disc->type_label);
		} else {
			mvwprintw(cdw_ui.disc_info_view.subwindow, row_h, pcol,
				  /* 2TRANS: this is string displaying information
				     about disc: 1st %s is disc type (e.g. CD-RW),
				     2nd %s is file system type on a disc; resulting
				     string may look like "CD-RW / ISO9660"; keep short */
				  _("%s / %s"), disc->type_label, disc->cdio->ofs->type_label);
		}
	}


	/* empty? */
	/* 2TRANS: 'Empty' means that there is no data on disc;
	   'Y', 'N' or '?' will follow after colon; keep short */
	mvwprintw(cdw_ui.disc_info_view.subwindow, row_e, pcol, _("Empty: "));
	mvwprintw(cdw_ui.disc_info_view.subwindow, row_e, vcol, _("%s"),
		  cdw_utils_get_cdw_bool_type_char(disc->state_empty));

	/* writable? */
	/* 2TRANS: 'Writable' means that it is possible to write to disc;
	   'Y', 'N' or '?' will follow after colon; keep short */
	mvwprintw(cdw_ui.disc_info_view.subwindow, row_w, pcol, _("Writable: "));
	if (disc->type_writable == CDW_TRUE) {
		mvwprintw(cdw_ui.disc_info_view.subwindow, row_w, vcol, _("%s"),
			  cdw_utils_get_cdw_bool_type_char(disc->state_writable));
	} else {
		mvwprintw(cdw_ui.disc_info_view.subwindow, row_w, vcol, _("%s"),
			  cdw_utils_get_cdw_bool_type_char(disc->type_writable));
	}

	/* erasable? */
	/* 2TRANS: "Erasable" means: 'content can be blanked/erased/wiped out'
	   'Y' or 'N' will follow after colon; keep short */
	mvwprintw(cdw_ui.disc_info_view.subwindow, row_e2, pcol, _("Erasable: "));
	mvwprintw(cdw_ui.disc_info_view.subwindow, row_e2, vcol, _("%s"),
		  cdw_utils_get_cdw_bool_type_char(disc->type_erasable));

	/* a special hack until cdw fully supports DVD+R DL */
#if 0   /* this code should probably be removed as this kind of decisions
	   shouldn't be made at this level */
	if (disc->type == CDW_DVD_RP_DL
	    && config.support_dvd_rp_dl) {

		const char *label_unknown = cdw_utils_get_cdw_bool_type_char(CDW_UNKNOWN);

		mvwprintw(cdw_ui.disc_info_view.subwindow, row_e, vcol, _("%s"), label_unknown);
		mvwprintw(cdw_ui.disc_info_view.subwindow, row_w, vcol, _("%s"), label_unknown);
		mvwprintw(cdw_ui.disc_info_view.subwindow, row_e2, vcol, _("%s"), label_unknown);
	}
#endif

	wattrset(cdw_ui.disc_info_view.subwindow, A_NORMAL | COLOR_PAIR(CDW_COLORS_MAIN));
	wrefresh(cdw_ui.disc_info_view.subwindow);

	return;
}





/**
   Refresh part of main application window.
*/
void cdw_main_ui_main_window_wrefresh_part(int n_lines, int n_cols, int begin_y, int begin_x)
{
	if (cdw_ui.window == (WINDOW *) NULL) {
		/* previously there was an assert() here, but this function
		   may be called by dialogbox when main ui (and thus main
		   window) is not initialized yet; example: when after starting
		   cdw a path to log file is checked and it is reset,
		   main window is still not created, but cdw displays dialog
		   that informs user about resetting path, and the dialog
		   window calls this function */
		cdw_vdm ("ERROR: called the function for null window\n");
		return;
	}

	WINDOW *window = derwin(cdw_ui.window, n_lines, n_cols, begin_y, begin_x);
	if (window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create derived window for refresh\n");
	} else {
		wrefresh(window);
		cdw_window_delete(&window);
	}

	return;
}





/**
   \brief Redraw content of main app window
*/
void cdw_main_ui_main_window_wrefresh(void)
{
	cdw_assert (cdw_ui.window, "called the function for null window\n");

	redrawwin(cdw_ui.window);
	wrefresh(cdw_ui.window);

	return;
}





cdw_rv_t cdw_main_window_volume_info_view_update(long long disc_sectors_used, long long disc_sectors_total, bool rescan_selected_files, bool follow_symlinks)
{
	static long long disc_capacity_sectors_used = 0;
	static long long disc_capacity_sectors_total = 0;

	if (disc_sectors_total >= 0) {
		disc_capacity_sectors_total = disc_sectors_total;
		if (disc_sectors_used >= 0) {
			disc_capacity_sectors_used = disc_sectors_used;
		}
	}

	size_t n_files = cdw_file_manager_number_of_selected_files();
	cdw_sdm ("INFO: number of selected files = %zd\n", n_files);

	bool error = false;

	double selected_files_size_mb = 0.0;
	if (n_files != 0) {
		if (rescan_selected_files) {
			selected_files_size_mb = cdw_file_manager_calculate_selected_files_size_mb(follow_symlinks);
			if (selected_files_size_mb < 0.0) {
				error = true;
			}
		} else {
			long long int size = cdw_selected_files_get_size();
			selected_files_size_mb = (((double) size) / 1024.0) / 1024.0;
		}
	}

	int total_space = 0;
	double used_space =  0.0;
	bool show_disc_info = false;

	if (global_config.volume_size_id == CDW_CONFIG_VOLUME_SIZE_AUTO) {
		used_space = (double) disc_capacity_sectors_used / 512.0;
		total_space = (int) disc_capacity_sectors_total / 512;

		if (total_space > 0) {
			show_disc_info = true;
		} else {
			/* some software/disc configurations may result
			   in total_space == 0, e.g. I can't figure how to
			   read DVD disc capacity from wodim's output;
			   in such cases volume size window will behave like
			   for "custom volume size" or "fixed volume size" */
		}
	} else {
		/* either custom value, or constant value corresponding
		   to total capacity of some preselected disc type */
		total_space = (int) cdw_config_get_current_volume_size_value();
	}

	cdw_main_ui_volume_info_view_display_data_new(show_disc_info, n_files, used_space, selected_files_size_mb, total_space);

	if (error) {
		cdw_vdm ("ERROR: size of selected files is < 0\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("One or more of selected files may be missing. Check log file for more information - press 'L' key in main window."),
				   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		cdw_main_ui_main_window_wrefresh();
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Display disc usage summary at the bottom of cdw window

   Display following info at the bottom of main UI window:
    - disc capacity and usage
    - number and size of new, selected files
    - amount of free space left on disc or in target volume
    - bar indicator and percentage value of free space left

   Disc capacity and usage is displayed only if \p show_disc_info
   is true (and the argument should be true if item selected
   from "ISO volume size" dropdown in configuration window is
   "Get sizes from disc").

   The function does not collect nor calculate any data (except for some
   local variables), it just uses current values from some global variables.

   \param show_disc_info - controls if information about disc capacity
                           and usage should be displayed
   \param n_files - number of files selected for burning
   \param used_space - amount of space already used on optical disc
   \param selected_size - size of selected files
   \param total_space - total capacity of optical disc
*/
void cdw_main_ui_volume_info_view_display_data_new(bool show_disc_info, size_t n_files, double used_space, double selected_size, int total_space)
{
	cdw_assert (cdw_ui.volume_info_view.subwindow != (WINDOW *) NULL,
		    "ERROR: called the function for NULL subwindow\n");
	WINDOW *subwindow = cdw_ui.volume_info_view.subwindow;
	wattrset(subwindow, A_BOLD | COLOR_PAIR(CDW_COLORS_MAIN));

	int bar_row = 4;    /* usage bar row */
        const int col = 2;        /* start column for content drawn in window */
	const int bar_width = 30; /* number of cells in bar, does not include borders */

	/* clean row in which usage bar and information about space usage is displayed */
	mvwhline(subwindow, 1, 1, ' ', cdw_ui.volume_info_view.n_cols - 2);
	mvwhline(subwindow, 2, 1, ' ', cdw_ui.volume_info_view.n_cols - 2);
	mvwhline(subwindow, 3, 1, ' ', cdw_ui.volume_info_view.n_cols - 2);
	mvwhline(subwindow, bar_row, 1, ' ', cdw_ui.volume_info_view.n_cols - 2);
	redrawwin(subwindow);
	wrefresh(subwindow);

	/* TODO: implement displaying size in kB or GB when applicable;
	   this won't be simple as it will require changes in few
	   different places */
	mvwprintw(subwindow, 2, col, n_files ?
		  /* 2TRANS: '%d' is a number of new files selected for
		     burning, '%.1f' is a size of the files; keep tilde
		     ('~') char or equivalent, as exact size of selected
		     files is not 100% precise */
		  _("Selected files (%d): ~%.1f MB") :
		  /* 2TRANS: '%d' is a number of new files selected for
		     burning, '%.1f' is a size of the files; */
		  _("Selected files (%d): %.1f MB"),
		  n_files, selected_size);

	if (total_space > 0) {
		if (show_disc_info) {
			/* 2TRANS: this is information about disc usage
			   and capacity: '%.1f' is size of data on disc,
			   '%d' is a capacity (total size) of disc; keep
			   tilde ('~') char or equivalent, as exact sizes
			   are not 100% precise */
			mvwprintw(subwindow, 1, col, _("On disc: ~%.1f / ~%d MB"),
				  used_space, total_space);
		}

		/* left and right border of usage bar */
		mvwaddch(subwindow, bar_row, col, '[');
		mvwaddch(subwindow, bar_row, col + bar_width + 1, ']');

		double percent_used = used_space * 100.0 / (double) total_space;
		double percent_selected = selected_size * 100.0 / (double) total_space;
		int n = (int) ((percent_used * (double) bar_width) / 100);
		int m = 0;
		double free_space = (double) total_space - used_space - selected_size;
		if (free_space > 0.0) {
			m = (int) ((percent_selected * (double) bar_width) / 100);
			if (percent_used + percent_selected > 90.0) {
				/* 90%: arbitrary value of 'safe' level of disc usage */
				wattrset(subwindow, COLOR_PAIR(CDW_COLORS_WARNING));
			}
			/* 2TRANS: this is information about available
			   space left on disc; keep tilde ('~') char
			   or equivalent */
			mvwprintw(subwindow, 3, col, _("Free space: ~%.1f MB"), free_space);
		} else {
			/* files size (already burned  + newly selected) is larger than capacity of disc */
			m = bar_width - n;
			wattrset(subwindow, COLOR_PAIR(CDW_COLORS_ERROR));
			/* 2TRANS: this is information about available
			   space left on disc: 0 MB, i.e. no space left */
			mvwprintw(subwindow, 3, col, _("Free space: 0 MB"));
		}
		wattrset(subwindow, COLOR_PAIR(CDW_COLORS_MAIN));

		mvwhline(subwindow, bar_row, col + 1, ACS_BLOCK, n); /* +1 to skip left brace [ */
		mvwhline(subwindow, bar_row, col + 1 + n, '+', m); /* +1 to skip left brace [ */
		mvwprintw(subwindow, bar_row, col + bar_width + 3, /* 3 to move past indicator bar */
			  /* 2TRANS: this is information about usage of disc space:
			     '%2.1f%%' is information about used space, displayed
			     as percentage of total disc size; keep tilde ('~') char
			     or equivalent */
			  percent_used + percent_selected > 0.1 ?  _("~%2.1f%%") : _("%2.1f%%"),
			  percent_used + percent_selected);
	} else {
		/* no information about total disc size; assume that size
		   of files on disc is also unknown; the only thing that
		   we know is size of selected files; information about
		   new files has been already printed */
	}

	wattrset(subwindow, A_NORMAL | COLOR_PAIR(CDW_COLORS_MAIN));
	wrefresh(subwindow);

	return;
}





/**
   \brief Delete selected files, do necessary updates

   This function checks if there are any files to delete from list of
   selected files, and calls selected_files_delete() to allow user
   to delete files from the list.

   If selected_files_delete() was called, existing list of graftpoints
   is deleted because it is outdated (new one should be created only
   when attempting to do sth using selected files).
*/
cdw_rv_t cdw_main_ui_delete_from_selected_files(void)
{
        size_t num = cdw_file_manager_number_of_selected_files();

	if (num > 0) { /* if cd files list not empty */
		/* first set_menu_fore() is used to un-highlight main menu
		   'Delete' button when focus is in file selection area */
		set_menu_fore(cdw_ui.main_menu.menu, COLOR_PAIR(CDW_COLORS_MENU));
		wrefresh(cdw_ui.main_menu.subwindow);

		cdw_file_manager_handle_deleting_from_selected_files();

		set_menu_fore(cdw_ui.main_menu.menu, COLOR_PAIR(CDW_COLORS_MENU) | A_REVERSE);
		wrefresh(cdw_ui.main_menu.subwindow);
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("No files to delete"),
				   /* 2TRANS: this is message in dialog window. User
				      wants to deselect files from list, but there
				      are no files at this time */
				   _("Nothing to delete - no files selected."),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
	}

	return CDW_OK;
}





cdw_rv_t cdw_main_ui_add_to_selected_files(void)
{
	cdw_file_manager_handle_adding_to_selected_files();

	return CDW_OK;
}





/**
   Create subwindow in which information about selected files is displayed.
   Display initial view in this subwindow.

   \return CDW_OK on success
   \return CDW_ERROR if some error occurred
*/
cdw_rv_t cdw_main_ui_volume_info_view_create(void)
{
	cdw_assert (cdw_ui.window, "calling the function for null main window\n");

	cdw_ui.volume_info_view.subwindow = derwin(cdw_ui.window,
						   cdw_ui.volume_info_view.n_rows,
						   cdw_ui.volume_info_view.n_cols,
						   cdw_ui.volume_info_view.begin_y,
						   cdw_ui.volume_info_view.begin_x);

	if (!cdw_ui.volume_info_view.subwindow) {
		cdw_vdm ("ERROR: failed to create subwindow with derwin() for selected files info view\n");
		return CDW_ERROR;
	}

	cdw_window_add_strings(cdw_ui.volume_info_view.subwindow,
			       /* 2TRANS: this is title of area showing basic info (total
				  size, CD usage) about selected files */
			       _("ISO volume"), (char *) NULL);

	return CDW_OK;
}





/**
   \brief Show program license

   Try to display GPL-2 file from system-default location, if this fails
   then display basic license info in dialog box.
*/
void cdw_main_ui_show_license(void)
{
	size_t len1 = strlen(COMMON);
	size_t len2 = strlen("/GPL-2");
	char *fullpath = (char *) malloc(len1 + len2 + 1);

	bool success = false;
	if (fullpath) {
		/* try to display license from /usr/share/common-licenses/ */
		snprintf(fullpath, len1 + len2 + 1, "%s/GPL-2", COMMON);
		if (!access(fullpath, R_OK)) {
			/* display license information from standard file provided by OS */
			/* 2TRANS: this is title of dialog window showing cdw license */
			cdw_rv_t crv = cdw_text_file_viewer_with_fullpath(fullpath, _("License"));
			if (crv == CDW_OK) {
				success = true;
			}
		}

		free(fullpath);
		fullpath = (char *) NULL;
	}

	if (!success) {
		/* this text turned out to be too long for passing it
		   directly as function's argument */
		/* 2TRANS: this is standard information about licensing
		   program under GPL; it may be non-translatable in your area */
		char *msg = strdup(_("This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.\n\nThis program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.\n\nYou should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA"));
		if (msg) {
			/* 2TRANS: this is title of dialog window, in which
			   basic information about licensing is displayed */
			cdw_buttons_dialog(_("cdw license"), msg,
					   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
			free(msg);
			msg = (char *) NULL;
		} else {
			/* 2TRANS: this is title of dialog window, in which
			   basic information about licensing is displayed */
			cdw_buttons_dialog(_("cdw license"),
					   /* 2TRANS: this is short information
					      about licensing program under GPL;
					      it may be non-translatable in your area */
					   _("This program is distributed under terms of GNU General Public License either version 2, or (at your option) any later version. This program is distributed WITHOUT ANY WARRANTY."),
					   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		}

	}

	return;
}




cdw_rv_t cdw_main_ui_tooltips_view_create(void)
{
	cdw_assert (cdw_ui.window, "calling the function for null main window\n");

	cdw_ui.tooltips_view.subwindow = derwin(cdw_ui.window,
						cdw_ui.tooltips_view.n_rows,
						cdw_ui.tooltips_view.n_cols,
						cdw_ui.tooltips_view.begin_y,
						cdw_ui.tooltips_view.begin_x);

	if (!cdw_ui.tooltips_view.subwindow) {
		cdw_vdm ("ERROR: failed to create subwindow with derwin() for tooltip view\n");
		return CDW_ERROR;
	} else {
		wbkgd(cdw_ui.tooltips_view.subwindow, A_NORMAL | COLOR_PAIR(CDW_COLORS_MAIN));
		wattrset(cdw_ui.tooltips_view.subwindow, A_BOLD | COLOR_PAIR(CDW_COLORS_TOOLTIPS));

		return CDW_OK;
	}
}





WINDOW *cdw_main_ui_get_tooltips_subwindow(void)
{
	cdw_assert(cdw_ui.tooltips_view.subwindow,
		   "called help tooltip before initializing cdw ui -> tooltips view\n");
	return cdw_ui.tooltips_view.subwindow;
}





MENU *cdw_main_ui_get_main_menu(void)
{
	cdw_assert(cdw_ui.main_menu.menu,
		   "called help tooltip before initializing cdw ui -> menu\n");
	return cdw_ui.main_menu.menu;
}





WINDOW *cdw_main_ui_get_main_window(void)
{
	cdw_assert(cdw_ui.window,
		   "called help tooltip before initializing cdw ui -> window\n");

	return cdw_ui.window;
}





void cdw_main_ui_init_values(void)
{
	cdw_ui.tooltips_view.subwindow = (WINDOW *) NULL;
	cdw_ui.disc_info_view.subwindow = (WINDOW *) NULL;
	cdw_ui.volume_info_view.subwindow = (WINDOW *) NULL;
	cdw_ui.files_list_view.subwindow = (WINDOW *) NULL;
	cdw_ui.main_menu.menu = (MENU *) NULL;
	cdw_ui.main_menu.items = (ITEM **) NULL;
	cdw_ui.main_menu.subwindow = (WINDOW *) NULL;
	cdw_ui.main_menu.menu_subwindow = (WINDOW *) NULL;
	cdw_ui.window = (WINDOW *) NULL;

	/* 'sizes' are layout constraints and parameters */
	cdw_main_ui_recalculate_sizes();

	return;
}





void cdw_main_ui_handle_follow_symlinks_change(bool old_state, bool current_state)
{
	if (current_state != old_state) {
		/* user changed the way how symlinks should be treated,
		   and this may affect size of selected files */
		cdw_vdm ("INFO: changed \"follow symlinks\" to %s, recalculating files size\n",
			 current_state ? "true" : "false");
		cdw_main_window_volume_info_view_update(-1, -1, true, current_state);
		cdw_file_manager_regenerate_selected_files_view();
	} else {
		cdw_vdm ("INFO: \"follow symlinks\" is still %s\n", current_state ? "true" : "false");
	}
	return;
}





/* *** unused code *** */

#if 0

/**
   \brief Display CD usage summary on bottom of cdw window

   Display amount of space used by selected files, amount of
   space that would stay free on disk, and percent of CD disk usage.
   CD/DVD size is taken from config panel/file.

   I try to pack all information in just two lines, so that it is always
   visible and always updated when file selector is on top and files_info area
   is partially covered by file selector.

   The function does not collect nor calculate any data (except for some local
   variables), it just uses current values from some global variables.
*/
void cdw_main_ui_volume_info_view_display_data(size_t n_files, double size_mb)
{
	cdw_assert (cdw_ui.volume_info_view.subwindow != (WINDOW *) NULL,
		    "called the function for null subwindow\n");
	WINDOW *subwindow = cdw_ui.volume_info_view.subwindow;
	wattrset(subwindow, A_BOLD | COLOR_PAIR(CDW_COLORS_MAIN));

	int s_row = 3;    /* "Size: " row */
	int b_row = 4;    /* usage bar row */
        const int col = 2;        /* start column for content drawn in window */
	const int bar_width = 30; /* number of cells in bar, does not include borders */

	/* clean row in which usage bar and information about space usage is displayed */
	mvwhline(subwindow, s_row, 1, ' ', cdw_ui.volume_info_view.n_cols - 2);
	mvwhline(subwindow, b_row, 1, ' ', cdw_ui.volume_info_view.n_cols - 2);

	/* left and right border of usage bar */
	mvwaddch(subwindow, b_row, col, '[');
	mvwaddch(subwindow, b_row, col + bar_width + 1, ']');

	bool too_much = false;
	if (size_mb > (double) global_config.volume_size_value) {
		too_much = true;
		wattrset(subwindow, COLOR_PAIR(CDW_COLORS_ERROR));
	} else {
		/* 0.8 (80%): arbitrary value of 'safe' level of disc usage */
		if (size_mb <= ((double) global_config.volume_size_value * 0.8)) {
			wattrset(subwindow, COLOR_PAIR(CDW_COLORS_MAIN));
		} else {
			wattrset(subwindow, COLOR_PAIR(CDW_COLORS_WARNING));
		}
	}

	double percent = size_mb * 100 / ((double) global_config.volume_size_value);
	int n = 0;
	if (too_much) { /* files size is larger than available space */
		/* 2TRANS: this string displays total size of dirs/files
		   selected ('%.1f'), total available space ('%d') and
		   number of items selected (second '%d'). */
		mvwprintw(subwindow, s_row, col, _("%.1f/%ld MB in %d items"),
			  size_mb, global_config.volume_size_value, n_files);

		n = bar_width;
	} else {
		/* 2TRANS: this string displays total size of dirs/files
		   selected ('%.1f'), total available space ('%d'), number of
		   items selected (second '%d'), and free space left ('%.0f'). */
		mvwprintw(subwindow, s_row, col, _("%.1f/%ld MB in %d items, %.0f MB free"),
			  size_mb, global_config.volume_size_value,
			  n_files, (double) global_config.volume_size_value - size_mb);

		n = (int) ((percent * (double) bar_width) / 100);
	}

	mvwhline(subwindow, b_row, col + 1, ACS_BLOCK, n); /* +1 to skip left brace [ */

	mvwprintw(subwindow, b_row, col + bar_width + 3, /* 3 to move past indicator bar */
		  /* 2TRANS: this is information about usage of CD space:
		     '%2.1f%%' is information about used space, displayed
		     as percentage of total disc size */
		  _("%2.1f%%"), percent);

	wattrset(subwindow, A_NORMAL | COLOR_PAIR(CDW_COLORS_MAIN));
	wrefresh(subwindow);

	return;
}

#endif

