package charactermanaj.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.sql.Timestamp;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

/**
 * ファイルに保存されるPropertiesアクセスの実装.<br>
 * ファイルの読み込みまたは書き込みがあると、それ以降、排他制御される.<br>
 * ファイルを閉じた後は読み込み・書き込みはできず、プロパティの更新もできなく、IllegalStateException例外となります.<br>
 * ただし、閉じた後でもプロパティの取得は可能です.<br> 
 * @author seraphy
 */
public class FileMappedProperties extends AbstractMap<String, String> {

	/**
	 * データベースファイル
	 */
	private File file;
	
	/**
	 * データベースファイルへのランダムアクセス
	 */
	private RandomAccessFile accessFile;
	
	/**
	 * ランダムアクセスファイルへのファイルチャネル(排他制御のため)
	 */
	private FileChannel channel;
	
	/**
	 * プロパティ実体
	 */
	private Properties props = new Properties();
	
	/**
	 * 変更フラグ
	 */
	private boolean modified;
	
	
	/**
	 * データベースとなるファイルを指定してプロパティを構築する.<br>
	 * 呼び出された時点でファイルがなければ作成される.<br> 
	 * @param file ファイル
	 * @throws IOException 失敗
	 */
	public FileMappedProperties(File file) throws IOException {
		if (file == null) {
			throw new IllegalArgumentException();
		}
		this.file = file;
		this.accessFile = new RandomAccessFile(file, "rw");
	}
	
	/**
	 * データベースとなるファイル
	 * @return データベースとなるファイル
	 */
	public File getFile() {
		return file;
	}

	/**
	 * 変更されているか?<br>
	 * putまたはremoveによって設定される.<br>
	 * save、load、clearのいずれかによりリセットされる.<br>
	 * @return 変更されている場合はtrue
	 */
	public boolean isModified() {
		return modified;
	}
	
	/**
	 * データベースとなるファイルからプロパティを読み取り排他制御をかける.<br>
	 * 以降、closeされるまで、ファイルはロックされます.<br>
	 * すでに読み込まれている場合は何もしません。(ロックされているので自身以外の書き込みは想定しないため).<br>
	 * @throws IOException 失敗
	 */
	public void load() throws IOException {
		checkOpen();
		if (channel != null) {
			// すでにロード済みと見なす.
			return;
		}
		
		props.clear();
		channel = accessFile.getChannel();
		channel.lock(); // 全域に排他ロック
		
		int siz = (int) channel.size();
		if (siz == 0) {
			// 空なので読み込まない.
			return;
		}
		byte[] data = new byte[siz];
		ByteBuffer buf = ByteBuffer.wrap(data);
		channel.read(buf);
		
		InputStream bis = new ByteArrayInputStream(data);
		try {
			props.loadFromXML(bis);
		} finally {
			bis.close();
		}
	}
	
	@Override
	public void clear() {
		super.clear();
		modified = false;
	}
	
	/**
	 * 現在のプロパティの内容をファイルに書き戻します.<br>
	 * ファイルがロックされていない場合はロックされ、以降、closeされるまでロックを維持します.<br>
	 * 呼び出される都度、ファイルを一括更新します.<br>
	 * @throws IOException 失敗
	 */
	public void save() throws IOException {
		checkOpen();
		if (channel == null) {
			channel = accessFile.getChannel();
			channel.lock(); // 全域に排他ロック
		}
		
		channel.position(0); // 先頭に戻す.
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		try {
			String comment = file.getName().toString() + " "
					+ new Timestamp(System.currentTimeMillis());
			props.storeToXML(bos, comment);

		} finally {
			bos.close();
		}
		
		byte[] data = bos.toByteArray();
		ByteBuffer buf = ByteBuffer.wrap(data);
		channel.write(buf);
		channel.truncate(data.length);
		
		modified = false;
	}

	/**
	 * ファイルを閉じます.<br>
	 * ロックされている場合はロックが解除されます.<br>
	 * 未保存のプロパティは保存されません.<br>
	 * @throws IOException 失敗
	 */
	public void close() throws IOException {
		if (channel != null) {
			channel.close();
			channel = null;
		}
		if (accessFile != null) {
			accessFile.close();
			accessFile = null;
		}
	}
	
	@Override
	protected void finalize() throws Throwable {
		close(); // 念のため
	}
	
	@Override
	public String put(String key, String value) {
		checkOpen();
		String oldValue = (String) props.setProperty(key, value);
		
		// 変更チェック
		if (value != oldValue) {
			if (value == null || oldValue == null || !value.equals(oldValue)) {
				// 双方がnullでなく、いずれか一方がnullであるか、equalsが一致しない場合は変更あり.
				modified = true;
			}
		}
		return oldValue;
	}
	
	@Override
	public Set<Map.Entry<String, String>> entrySet() {
		final Set<Map.Entry<Object, Object>> entrySet = props.entrySet();
		return new AbstractSet<Map.Entry<String, String>>() {
			@Override
			public Iterator<Map.Entry<String, String>> iterator() {
				final Iterator<Map.Entry<Object, Object>> ite = entrySet.iterator();
				return new Iterator<Map.Entry<String, String>>() {
					public boolean hasNext() {
						return ite.hasNext();
					}
					public Map.Entry<String, String> next() {
						final Entry<Object, Object> entry = ite.next();
						return new Map.Entry<String, String>() {
							public String getKey() {
								return (String) entry.getKey();
							}
							public String getValue() {
								return (String) entry.getValue();
							}
							public String setValue(String value) {
								checkOpen();
								return (String) entry.setValue(value);
							}
						};
					}
					public void remove() {
						checkOpen();
						modified = true;
						ite.remove();
					}
				};
			}
			@Override
			public int size() {
				return entrySet.size();
			}
		};
	}
	
	protected void checkOpen() {
		if (accessFile == null) {
			throw new IllegalStateException("file is already closed." + file);
		}
	}
	
}
