﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Microsoft.AspNetCore.Razor.Language;
using Microsoft.CodeAnalysis.ExternalAccess.Razor.Features;
using Xunit;

namespace Microsoft.VisualStudio.Razor.LanguageClient.Cohost.Formatting;

public class CSharpSyntaxFormattingOptionsTest_Generator
{
    /// <summary>
    /// This test can be run manually to generate exhaustive tests for RazorCSharpSyntaxFormattingOptions.
    /// </summary>
    /// <remarks>
    /// After running this test, the file will be updated but you should expect the "expected" output to be incorrect
    /// and you'll have to manually adjust it, which is why regenerating the tests is a manual process.
    /// Or make the test generation much smarter I guess :)
    /// </remarks>
    [Fact(Skip = "This test needs to be run manually. Just remove this skip message, and run it if you want")]
    public void GenerateTests()
    {
        //lang=C#-test
        var fileHeader = """
            // <auto-generated />
            
            using System;
            using System.Threading.Tasks;
            using Microsoft.AspNetCore.Razor.Language;
            using Microsoft.AspNetCore.Razor.Test.Common;
            using Microsoft.CodeAnalysis.ExternalAccess.Razor.Features;
            using Microsoft.CodeAnalysis.Razor.Formatting;
            using Xunit;
            using Xunit.Abstractions;

            namespace Microsoft.VisualStudio.Razor.LanguageClient.Cohost.Formatting;

            [Collection(HtmlFormattingCollection.Name)]
            public class CSharpSyntaxFormattingOptionsTest(FormattingTestContext context, HtmlFormattingFixture fixture, ITestOutputHelper testOutput)
                : FormattingTestBase(context, fixture.Service, testOutput), IClassFixture<FormattingTestContext>
            {
            """;

        //lang=C#-test
        var fileFooter = """
            }
            """;

        //lang=C#-test
        var testTemplate = """"
            [FormattingTestFact]
            public async Task {TestName}()
            {
                await RunFormattingTestAsync(
                    input: """
                        @code
                        {
                        private bool IconMenuActive { get; set; } = false;
                        protected void ToggleIconMenu(bool iconMenuActive)
                        {
                        IconMenuActive = iconMenuActive;
                        }
                        }
                        """,
                    expected: """
                        @code
                        {
                            private bool IconMenuActive { get; set; } = false;
                            protected void ToggleIconMenu(bool iconMenuActive)
                            {
                                IconMenuActive = iconMenuActive;
                            }
                        }
                        """,
                    csharpSyntaxFormattingOptions: RazorCSharpSyntaxFormattingOptions.Default with
                    {
                        NewLines = {RazorNewLinePlacement},
                        WrappingKeepStatementsOnSingleLine = {WrappingKeepStatementsOnSingleLine},
                        WrappingPreserveSingleLine = {WrappingPreserveSingleLine},
                    });
            }
        """";

        bool[] booleanOptions = [true, false];

        var projectPath = TestProject.GetProjectDirectory(typeof(CSharpSyntaxFormattingOptionsTest_Generator), layer: TestProject.Layer.Tooling);
        var testFileName = Path.Combine(projectPath, """Cohost\Formatting\CSharpSyntaxFormattingOptionsTest.cs""");

        var addExtraLine = false;

        using var writer = new StreamWriter(testFileName);
        writer.WriteLine(fileHeader);

        foreach (var nl in GetAllValues(typeof(RazorNewLinePlacement)))
            foreach (var keep in booleanOptions)
                foreach (var preserve in booleanOptions)
                {
                    if (addExtraLine)
                    {
                        writer.WriteLine();
                    }

                    addExtraLine = true;

                    var testName = $"CSharpSyntaxFormattingOptions_Without{nl.without}_{(keep ? "SingleLine" : "MultiLine")}_{(preserve ? "PreserveSingle" : "DontPreserve")}";
                    writer.WriteLine(testTemplate
                        .Replace("{TestName}", testName)
                        .Replace("{RazorNewLinePlacement}", nl.value)
                        .Replace("{WrappingKeepStatementsOnSingleLine}", keep.ToString().ToLowerInvariant())
                        .Replace("{WrappingPreserveSingleLine}", preserve.ToString().ToLowerInvariant()));
                }

        writer.WriteLine(fileFooter);
    }

    private static IEnumerable<(string without, string value)> GetAllValues(Type enumType)
    {
        var names = Enum.GetNames(enumType).Where(n => n != "None").ToArray();

        foreach (var name in names)
        {
            yield return (name, string.Join(" | ", names.Where(n => n != name).Select(n => $"{enumType.Name}.{n}")));
        }
    }
}
