/* 
 * mask_rectangle.cc -- ePiX rectangle crop mask
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.10
 * Last Change: August 09, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <cmath>

#include "functions.h"
//#include "errors.h"

#include "constants.h"

#include "pairs.h"

#include "Color.h"

#include "paint_style.h"
#include "length.h"

#include "pen_line.h"
#include "pen_fill.h"

#include "edge_data.h"
#include "screen_crop.h"

#include "screen_mask.h"
#include "mask_rectangle.h"

namespace ePiX {

  mask_rectangle::mask_rectangle(const pair& arg1, const pair& arg2)
    : m_hmin(min(arg1.x1(), arg2.x1())),
      m_hmax(max(arg1.x1(), arg2.x1())),
      m_vmin(min(arg1.x2(), arg2.x2())),
      m_vmax(max(arg1.x2(), arg2.x2()))
  {
    m_border.push_back(edge2d(br(), tr(), true));
    m_border.push_back(edge2d(tr(), tl(), true));
    m_border.push_back(edge2d(tl(), bl(), true));
    m_border.push_back(edge2d(bl(), br(), true));
  }

  mask_rectangle* mask_rectangle::clone() const
  {
    return new mask_rectangle(*this);
  }

  // corners and midpoints
  double mask_rectangle::h_min() const { return m_hmin; }
  double mask_rectangle::v_min() const { return m_vmin; }

  double mask_rectangle::h_max() const { return m_hmax; }
  double mask_rectangle::v_max() const { return m_vmax; }

  double mask_rectangle::h_avg() const { return 0.5*(m_hmin+m_hmax); }
  double mask_rectangle::v_avg() const { return 0.5*(m_vmin+m_vmax); }

  double mask_rectangle::h_size() const { return m_hmax - m_hmin; }
  double mask_rectangle::v_size() const { return m_vmax - m_vmin; }

  // rectangle-specific functions
  bool mask_rectangle::crops(const pair& arg) const
  {
    const double x(arg.x1());
    const double y(arg.x2());
    return !(   (m_hmin <= x) && (x <= m_hmax)
	     && (m_vmin <= y) && (y <= m_vmax));
  }

  std::list<edge2d>& mask_rectangle::crop_path(std::list<edge2d>& L) const
  {
    return crop_path_2nd(m_border, L);
  }

  std::list<edge2d>& mask_rectangle::crop_loop(std::list<edge2d>& L) const
  {
    return crop_loop_2nd(m_border, L);
  }

  pen_line mask_rectangle::border(const Color& col, const length& len) const
  {
    return pen_line(pen_data(col, len), Xfine(), m_border);
  }

  pen_fill mask_rectangle::backing(const Color& col) const
  {
    return pen_fill(col, pen_data(col, PLAIN_WIDTH), 0, m_border);
  }

  pen_line mask_rectangle::border() const
  {
    return pen_line(the_paint_style().line_pen(),
		    the_paint_style().base_pen(), m_border);
  }

  pen_fill mask_rectangle::backing() const
  {
    const Color& col(the_paint_style().fill_color());
    return pen_fill(col, pen_data(col, PLAIN_WIDTH), 0, m_border);
  }
} // end of ePiX namespace
