/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gnome-scan
 * Copyright (C) Étienne Bersac 2007 <bersace03@laposte.net>
 * 
 * gnome-scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

/**
 * SECTION: 
 * @stability: Unstable
 *
 * Base class for writing plugin to #GnomeScanPreviewArea. It's the
 * #GnomeScanParamWidget for #GnomeScanPreviewArea. It allow to define a list
 * of buttons added to the preview area, and changing a parameter value.
 *
 * This part is really supposed to change in the future, maybe by simply
 * subclassing GtkBin or something similar.
 **/

#include <glib/gi18n.h>
#include <string.h>
#include "gnome-scan-param-specs.h"
#include "gnome-scan-preview-plugin.h"
#include "gnome-scan-preview-area.h"
#include "gnome-scan-utils.h"

static void	gnome_scan_preview_plugin_class_init	(GnomeScanPreviewPluginClass *klass);
static void	gnome_scan_preview_plugin_init			(GnomeScanPreviewPlugin *plugin);

GType
gnome_scan_preview_plugin_get_type ()
{
	static GType type = 0;
	static GTypeInfo	tinfo = {
		sizeof (GnomeScanPreviewPluginClass),
		NULL, NULL, /* base */
		(GClassInitFunc) gnome_scan_preview_plugin_class_init, NULL, NULL, /* class */
		sizeof (GnomeScanPreviewPlugin), 0,
		(GInstanceInitFunc) gnome_scan_preview_plugin_init, NULL
	};
	static GTypeFundamentalInfo	finfo = {
		G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE
	};
	
	if (type == 0) {
		type = g_type_fundamental_next();
		g_type_register_fundamental (type,
									 "GnomeScanPreviewPlugin",
									 &tinfo,
									 &finfo,
									 G_TYPE_FLAG_ABSTRACT);
	}
	return type;
}

static void
gnome_scan_preview_plugin_class_init (GnomeScanPreviewPluginClass *klass)
{
}

static void
gnome_scan_preview_plugin_init (GnomeScanPreviewPlugin *plugin)
{
}

GType
gnome_scan_preview_plugin_type_register_static (const gchar *name,
												const GnomeScanPreviewPluginTypeInfo *pinfo)
{
	GTypeInfo info = {
		sizeof (GnomeScanPreviewPluginClass),
		NULL, NULL, /* base */
		(GClassInitFunc) pinfo->class_init, NULL, pinfo,
		pinfo->instance_size, 0,
		NULL, NULL,
	};
	
	return g_type_register_static (GNOME_TYPE_SCAN_PREVIEW_PLUGIN,
								   name, &info, 0);
}

static void
gspp_settings_changed (GnomeScanSettings *settings,
					   const gchar*name,
					   GnomeScanPreviewPlugin *gspp)
{
	GnomeScanPreviewPluginClass *klass = GNOME_SCAN_PREVIEW_PLUGIN_GET_CLASS (gspp);
	gboolean watched;
	gint i, n;
	
	watched = g_str_equal (name, g_param_spec_get_name (gspp->pspec));
	if (!watched) {
		n = g_strv_length (klass->watch);
		for (i = 0; i < n; i++) {
			watched = g_str_equal(name, klass->watch[i]);
			if (watched)
				continue;
		}
	}
	
	if (watched) {
		klass->changed (gspp);
		gnome_scan_preview_area_update (GNOME_SCAN_PREVIEW_AREA (gspp->preview_area), NULL);
	}
}

/**
 * gnome_scan_preview_plugin:
 * @type:			subtype of #GnomeScanPreviewPlugin to instanciate
 * @scan_plugin:	a #GnomeScanPlugin
 * @pspec:			the #GSParamSpec to manage
 * @gspa:			the #GnomeScanPreviewArea to extend
 * @settings:	    the #GnomeScanSettings where to store value
 * @box:			the box where to put buttons
 *
 * Returns: a #GnomeScanPreviewPlugin
 **/
GnomeScanPreviewPlugin*
gnome_scan_preview_plugin (GType type,
						   GnomeScanPlugin *scan_plugin,
						   GParamSpec *pspec,
						   GtkWidget *gspa,
						   GnomeScanSettings *settings,
						   GtkBox *box)
{
	g_return_val_if_fail (g_type_is_a (type, GNOME_TYPE_SCAN_PREVIEW_PLUGIN), NULL);
	GnomeScanPreviewPlugin* plugin = GNOME_SCAN_PREVIEW_PLUGIN (g_type_create_instance (type));
	GnomeScanPreviewPluginClass *klass = GNOME_SCAN_PREVIEW_PLUGIN_GET_CLASS (plugin);
	
	plugin->plugin			= scan_plugin;
	plugin->preview_area 	= gspa;
	plugin->settings		= g_object_ref (settings);
	plugin->pspec			= pspec;
	
	if (!gnome_scan_settings_get (settings, g_param_spec_get_name (pspec))) {
		gnome_scan_settings_init_value (plugin->settings,
										plugin->pspec);
	}
	
	klass->init (plugin);
	
	plugin->settings_changed =
		g_signal_connect (settings, "changed",
						  G_CALLBACK (gspp_settings_changed),
						  plugin);

	plugin->preview_changed =
		g_signal_connect_swapped (plugin->preview_area, "preview-received",
								  G_CALLBACK (klass->changed),
								  plugin);
	klass->build_ui (plugin, box);
	
	return plugin;
}

/**
 * gnome_scan_preview_plugin_freeze:
 * @gspp:   a #GnomeScanPreviewPlugin
 *
 * Freeze the plugin so that it doesn't react on value changes.
 **/
void
gnome_scan_preview_plugin_freeze(GnomeScanPreviewPlugin *gspp)
{
	gspp->frozen = TRUE;
}

/**
 * gnome_scan_preview_plugin_thaw:
 * @gspp:   a #GnomeScanPreviewPlugin
 *
 * Thaw the plugin so that it react on value changes.
 **/
void
gnome_scan_preview_plugin_thaw(GnomeScanPreviewPlugin *gspp)
{
	gspp->frozen  = FALSE;
}

/**
 * gnome_scan_preview_plugin_changed:
 * @gspp:   a #GnomeScanPreviewPlugin
 *
 * To be called when the preview has been changed.
 **/
void
gnome_scan_preview_plugin_changed (GnomeScanPreviewPlugin *gspp)
{
	if (!gspp->frozen) {
		GnomeScanPreviewPluginClass *klass = GNOME_SCAN_PREVIEW_PLUGIN_GET_CLASS (gspp);
		klass->changed(gspp);
	}
}

/**
 * gnome_scan_preview_plugin_draw_buffer:
 * @gspp:   a #GnomeScanPreviewPlugin
 * @gspa:   a #GnomeScaPreviewArea
 * @cr:		a cairo context
 *
 * For use by #GnomeScanPreviewArea
 *
 * Stability: Private
 **/
void
gnome_scan_preview_plugin_draw_buffer (GnomeScanPreviewPlugin *gspp,
									   GtkWidget *gspa,
									   cairo_t *cr)
{
	GnomeScanPreviewPluginClass *klass = GNOME_SCAN_PREVIEW_PLUGIN_GET_CLASS (gspp);
	klass->draw_buffer(gspp, gspa, cr);
}

/**
 * gnome_scan_preview_plugin_destroy:
 * @gspp:   the dying #GnomeScanPreviewPlugin
 **/
void
gnome_scan_preview_plugin_destroy	(GnomeScanPreviewPlugin *gspp)
{
	g_return_if_fail (GNOME_IS_SCAN_PREVIEW_PLUGIN (gspp));
	GnomeScanPreviewPluginClass *klass = GNOME_SCAN_PREVIEW_PLUGIN_GET_CLASS (gspp);
	g_signal_handler_disconnect (gspp->settings, gspp->settings_changed);
	g_signal_handler_disconnect (gspp->preview_area, gspp->preview_changed);
	g_object_unref (gspp->settings);
	klass->finalize (gspp);
	g_type_free_instance (&gspp->instance);
}

