/* 
 * gpsk31  - PSK31 for Linux with a GTK+ Interface
 * 
 * Copyright (C) 2002,2003,2004,2005  Thomas Ries <tries@gmx.net>
 * Copyright (C) 2008 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * The main author can be reached at pg4i@amsat.org or by smail-mail:
 * Joop Stakenborg, Bramengaarde 24, 3992KG Houten, The Netherlands.
 * 
 */

#include "socketif.h"

#include <stdio.h>
#include <string.h>

#include <errno.h>
#include <unistd.h>
#include <fcntl.h>

#include <gtk/gtk.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <signal.h>

/* global variables */
extern bool gpsk31debug;

/* module local variables */
static int listen_port=0;
static int listen_fd=0;
static int stream_fd=0;

/*
 * TCP socket connection
 */
void tcp_set_listen_port(int port){
   listen_port = port;
   tcp_listen();
   return;
}

/* prepare a listening TCP socket */
void tcp_listen(void) {
   struct sockaddr_in my_addr;
   int sts, on=1;
   int flags;

   /* disabled in configuration? */
   if (listen_port == 0) {
      listen_fd=-1;
      return;
   }

   /* ignore SIGPIPE of lost TCP connection */
   signal (SIGPIPE, SIG_IGN);

   memset(&my_addr, 0, sizeof(my_addr));
   my_addr.sin_family = AF_INET;
   my_addr.sin_port = htons(listen_port);

   listen_fd=socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
   if (gpsk31debug) g_message("listener on TCP port %i",listen_port);
   if (listen_fd < 0) {
      g_warning("socket returned error [%i:%s]",errno, strerror(errno));
      return;
   }

   if (setsockopt(listen_fd, SOL_SOCKET, SO_REUSEADDR, &on , sizeof(on)) < 0) {
      g_warning("setsockopt returned error [%i:%s]",errno, strerror(errno));
      return;
   }


   sts=bind(listen_fd, (struct sockaddr *)&my_addr, sizeof(my_addr));
   if (sts != 0) {
      g_warning("bind returned error [%i:%s]",errno, strerror(errno));
      close(listen_fd);
      listen_fd=-1;
      return;
   }

   /* set non-blocking */
   flags = fcntl(listen_fd, F_GETFL);
   if (flags < 0) {
      g_warning("fcntl returned error [%i:%s]",errno, strerror(errno));
      close(listen_fd);
      listen_fd=-1;
      return;
   }
   if (fcntl(listen_fd, F_SETFL, (long) flags | O_NONBLOCK) < 0) {
      g_warning("fcntl returned error [%i:%s]",errno, strerror(errno));
      close(listen_fd);
      listen_fd=-1;
      return;
   }

   listen (listen_fd, 1);
   return;
}

/* checks for new connection and does accept if capable (noone yet connected) */
void tcp_connect(void) {
   int sts;
   fd_set fdset;
   struct timeval timeout;
   int tmpfd;

   if (listen_fd <= 0) return;

   timeout.tv_sec=0;
   timeout.tv_usec=0;

   FD_ZERO(&fdset);
   FD_SET (listen_fd, &fdset);

   sts=select(listen_fd+1, &fdset, NULL, NULL, &timeout);
   if (sts > 0) {
      if (stream_fd != 0) {
         const char *msg="gpsk31: NOT-CONNECTED - other client connected, Sorry...\n";
         tmpfd=accept(listen_fd, NULL, NULL);
         sts = send(tmpfd, msg, strlen(msg), 0);
         close(tmpfd);
         if (gpsk31debug) g_message("Rejected new TCP connection, someone already connected.");
      } else {
         const char *msg="gpsk31: CONNECTED\n";
         stream_fd=accept(listen_fd, NULL, NULL);
         sts = send(stream_fd, msg, strlen(msg), 0);
         if (gpsk31debug) g_message("Accepted TCP connection [fd=%i]", stream_fd);
      }
   }

   return;
}

/* reads from TCP socket and return size & '\0' terminated string */
int tcp_read(char *buf, int buflen) {
   int sts;
   fd_set fdset;
   struct timeval timeout;
   int retlen=0;

   /* check the TCP connection and read stuff if present */
   if (stream_fd > 0) {
      timeout.tv_sec=0;
      timeout.tv_usec=0;

      FD_ZERO(&fdset);
      FD_SET (stream_fd, &fdset);

      sts=select(stream_fd+1, &fdset, NULL, NULL, &timeout);
      if (sts > 0) {
         sts = recv(stream_fd, buf, buflen-1, 0);
         /* got disconnected? */
         if (sts == 0) {
            close(stream_fd);
            if (gpsk31debug) g_message("Disconnected TCP connection [fd=%i]", stream_fd);
            stream_fd=0;
         } else {
            /* how many chars read from socket */
            buf[sts]='\0';
            retlen=sts;
         }
      }
   }

   return retlen;
}


int tcp_send(char *buf, int len) {
   int sts;
   sts = send(stream_fd, buf, len, MSG_MORE);
   return sts;
}

