/* 
Gregorio OpusTeX output format.
Copyright (C) 2007-2009 Elie Roux <elie.roux@telecom-bretagne.eu>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if ALL_STATIC == 0
    #include <gregorio/plugin.h>
#endif
#include <gregorio/struct.h>
#include <gregorio/unicode.h>
#include <gregorio/characters.h>
#include <gregorio/messages.h>

#include "opustex.h"

#if ALL_STATIC == 0
DECLARE_PLUGIN(opustex)
{
  .id = "otex",
  .name = "opustex",
  .description = "OpusTeX output plugin",
  .author = "Elie Roux <elie.roux@enst-bretagne.fr>",

  .file_extension = "tex",

  .type = GREGORIO_PLUGIN_OUTPUT,

  .write = write_score
};
#endif

int i;
int otexclef;
char loff = 0;
char centered = 0;
char italic = 0;
char key_change = 0;
char new_line = 0;

#if ALL_STATIC == 0
void
write_score (FILE * f, gregorio_score * score)
#else
void
opustex_write_score (FILE * f, gregorio_score * score)
#endif
{
  char first_syllable = 0;
  char clef_letter;
  int clef_line;
  gregorio_character *first_text;
  gregorio_syllable *current_syllable = score->first_syllable;

  if (!f) {
      gregorio_message (_
			   ("call with NULL file"),
			   "gregoriotex_write_score", ERROR, 0);
			   return;
  }

  if (score->number_of_voices != 1)
    {
      gregorio_message (_
			   ("opustex only works in monophony (for the moment)"),
			   "opustex_write_score", ERROR, 0);
    }

  fprintf(f, "%% File generated by gregorio %s\n", VERSION);
  fprintf (f,
	   "\\input opustex\n\\input opusgre\n\\input opuscho\n\n\\setgregorian1\n");
  // first we draw the initial (first letter) and the initial key
  if (score->first_voice_info)
    {
      gregorio_det_step_and_line_from_key (score->first_voice_info->
					      initial_key, &clef_letter,
					      &clef_line);
      if (clef_letter == 'f')
	{
	  fprintf (f, "\\setclefsymbol1\\gregorianFclef\n\\setclef1%d\n",
		   clef_line + 2);
	}
      else
	{
	  fprintf (f, "\\setclef1%d\n", clef_line);
	}
    }
  else
    {
      fprintf (f, "\\setclef13\n");
    }
  // a char that will contain 1 if it is the first syllable and 0 if not. It is for the initial.
  fprintf (f,
	   "\\musicindent10mm\n\\raisesong3\\Internote\n\\initiumgregorianum\n");
  first_text = gregorio_first_text (score);
  if (first_text)
    {
      fprintf (f, "\\musicinitial{}{");
      gregorio_write_initial (first_text, f,
				      (&otex_write_verb),
				      (&otex_print_char),
				      (&otex_write_begin),
				      (&otex_write_end),
				      (&otex_write_special_char));
      fprintf (f, "}%%\n");
      first_syllable = SKIP_FIRST_LETTER;
    }

  otexclef = score->first_voice_info->initial_key;
  current_syllable = score->first_syllable;
  while (current_syllable)
    {
      opustex_write_syllable (f, current_syllable,
					  &first_syllable);
      current_syllable = current_syllable->next_syllable;
    }
  fprintf (f, "\\bye\n");
}


char
opustex_is_out_of_neume (gregorio_syllable * syllable)
{
  if (!(syllable->text) && !(syllable->elements[1])
      && syllable->elements[0]->type != GRE_ELEMENT)
    {
      return 1;
    }
  return 0;
}

char
is_even (int c)
{
  if (c % 2 == 0)
    {
      return 1;
    }
  else
    {
      return 0;
    }
}

void
opustex_write_syllable (FILE * f, gregorio_syllable * syllable,
				    char *first_syllable)
{
  char finis = 0;
  char next_note;
  int nextposition = 0;

  gregorio_element *current_element = syllable->elements[0];
  if (opustex_is_out_of_neume (syllable))
    {
      if (current_element->type == GRE_BAR)
	{
	  if (syllable->next_syllable)
	    {
	      fprintf (f, "\\");
	      opustex_write_barline (f,
						 current_element->
						 element_type);
	      fprintf (f, "\n\\spatium\n");
	      current_element = current_element->next;
	    }
	  else
	    {
	      fprintf (f, "\\");
	      opustex_write_finis (f,
					       current_element->element_type);
	      fprintf (f, "\n");
	      finis = 1;
	    }
	  key_change = 0;
	  new_line = 0;
	  return;
	}
      if (current_element->type == GRE_SPACE)
	{
	  switch (current_element->element_type)
	    {
	    case SP_NO_SPACE:
	      fprintf (f, "\\nonspatium\n");
	      break;
	    case SP_ZERO_WIDTH:
	      fprintf (f, "\\Nonspatium\n");
	      break;
	    case SP_NEUMATIC_CUT:
	      fprintf (f, "\\spatiumparvum\n");
	      break;
	    case SP_LARGER_SPACE:
	      fprintf (f, " \\spatiumparvum\n");
	      break;
	    case SP_GLYPH_SPACE:
	      break;
	    default:
	      fprintf (f, "\\spatium\n");
	      break;
	    }
	  key_change = 0;
	  new_line = 0;
	  return;
	}
      next_note = find_next_note (current_element, syllable);
      if (syllable->next_syllable && syllable->next_syllable->elements[0]
	  && syllable->next_syllable->elements[0]->type == GRE_END_OF_LINE)
	{
	  new_line = 1;
	}
      if (current_element->type == GRE_C_KEY_CHANGE)
	{
	  if (next_note != 0)
	    {
	      otexclef =
		gregorio_calculate_new_key (C_KEY,
					       current_element->element_type -
					       48);
	      if (new_line == 1)
		{
		  fprintf (f, "\\loff{\\custos ");
		}
	      else
		{
		  fprintf (f, "\\CUSTOS ");
		}
	      opustex_print_note (f, next_note);
	      if (new_line == 1)
		{
		  fprintf (f, "}\n");
		}
	      fprintf (f,
		       "\\setclefsymbol1\\gregorianCclef\n\\setclef1%d\n",
		       current_element->element_type - 48);
	      if (new_line == 1)
		{
		  fprintf (f, "\\lineaproxima\n");
		}
	      else
		{
		  fprintf (f, "\\changeclefs\n");
		}
	    }
	  key_change = 1;
	  new_line = 0;
	  return;
	}
      if (current_element->type == GRE_F_KEY_CHANGE)
	{
	  if (next_note != 0)
	    {
	      otexclef =
		gregorio_calculate_new_key (F_KEY,
					       current_element->element_type -
					       48);
	      if (new_line == 1)
		{
		  fprintf (f, "\\loff{\\custos ");
		}
	      else
		{
		  fprintf (f, "\\CUSTOS ");
		}
	      opustex_print_note (f, next_note);
	      if (new_line == 1)
		{
		  fprintf (f, "}\n");
		}
	      fprintf (f,
		       "\\setclefsymbol1\\gregorianFclef\n\\setclef1%d\n",
		       current_element->element_type - 46);
	      if (new_line == 1)
		{
		  fprintf (f, "\\lineaproxima\n");
		}
	      else
		{
		  fprintf (f, "\\changeclefs\n");
		}
	    }
	  key_change = 1;
	  new_line = 0;
	  return;
	}
      if (current_element->type == GRE_END_OF_LINE)
	{
	  if (next_note == 0
	      || (syllable->next_syllable
		  && syllable->next_syllable->elements[0]
		  && (syllable->next_syllable->elements[0]->type ==
		      GRE_C_KEY_CHANGE
		      || syllable->next_syllable->elements[0]->type ==
		      GRE_F_KEY_CHANGE)) || key_change == 1)
	    {
	    }
	  else
	    {
	      fprintf (f, "\\custos ");
	      opustex_print_note (f, next_note);
	      fprintf (f, "\n\\lineaproxima\n");
	    }
	  key_change = 0;
	  new_line = 1;
	  return;
	}
    }
  else
    {
      if (syllable->next_syllable)
	{
	  nextposition = syllable->next_syllable->position;
	}
      fprintf (f, "\\sgn ");
      opustex_write_text (f, syllable->text, first_syllable);
      while (current_element)
	{
	  if (current_element->type == GRE_SPACE)
	    {
	      switch (current_element->element_type)
		{
		case SP_NO_SPACE:
		  fprintf (f, "\\nonspatium");
		  break;
		case SP_ZERO_WIDTH:
		  fprintf (f, "\\Nonspatium");
		  break;
		case SP_NEUMATIC_CUT:
		  fprintf (f, "\\spatiumparvum");
		  break;
		case SP_LARGER_SPACE:
		  fprintf (f, " \\spatiumparvum");
		  break;
		case SP_GLYPH_SPACE:
		  break;
		default:
		  fprintf (f, "\\spatium");
		  break;
		}
	      current_element = current_element->next;
	      key_change = 0;
	      new_line = 0;
	      continue;
	    }
	  if (current_element->type == GRE_BAR)
	    {
	      fprintf (f, "\\");
	      opustex_write_barline (f,
						 current_element->
						 element_type);
	      fprintf (f, "\\spatium");	// OpusTeX: inside a neume, divisio needs to be followed by a spatium
	      current_element = current_element->next;
	      key_change = 0;
	      new_line = 0;
	      continue;
	    }
	  if (current_element->type == GRE_TEXVERB_ELEMENT)
	    {
	      if (current_element->texverb)
	        {
	          fprintf (f, "%s", current_element->texverb);
	        }
	      current_element = current_element->next;
	      key_change = 0;
	      new_line = 0;
	      continue;
	    }
	  if (current_element->type == GRE_C_KEY_CHANGE
	      || current_element->type == GRE_F_KEY_CHANGE)
	    {
	      gregorio_message (_
				   ("clef change inside of a syllable doesn't work in OpusTeX"),
				   "opustex_write syllable",
				   ERROR, 0);
	      current_element = current_element->next;
	      continue;
	    }
	  if (current_element->type == GRE_END_OF_LINE)
	    {
	      if (current_element->next
		  && current_element->next->type == GRE_BAR)
		{
		  gregorio_message (_
				       ("line break cannot be placed before a divisio in OpusTeX"),
				       "opustex_write syllable",
				       ERROR, 0);
		}
	      else
		{
		  next_note =
		    find_next_note (current_element, syllable);
		  if (next_note == 0
		      || (!(current_element->next)
			  && (syllable->next_syllable
			      && syllable->next_syllable->elements[0]
			      && (syllable->next_syllable->elements[0]->
				  type == GRE_C_KEY_CHANGE
				  || syllable->next_syllable->elements[0]->
				  type == GRE_F_KEY_CHANGE)))
		      || key_change == 1)
		    {
		    }
		  else
		    {
		      fprintf (f, "\\custos ");
		      opustex_print_note (f, next_note);
		      fprintf (f, "\\lineaproxima");
		    }
		}
	      current_element = current_element->next;
	      key_change = 0;
	      new_line = 1;
	      continue;
	    }
	  opustex_write_element (f, current_element);
	  current_element = current_element->next;
	  key_change = 0;
	  new_line = 0;
	  continue;
	}
      if (loff >= 1)
	{
	  fprintf (f, "}");
	}
      loff = 0;
      fprintf (f, "\\egn\n");
      if (nextposition
	  && ((nextposition == WORD_BEGINNING)
	      || (nextposition == WORD_ONE_SYLLABLE)))
	{
	  fprintf (f, "\\spatium\n");
	}
    }
  if (!(syllable->next_syllable) && finis == 0)
    {
      fprintf (f, "\\Finisgregoriana\n");
    }
}


void
otex_write_begin (FILE * f, unsigned char style)
{
  switch (style)
    {
    case ST_ITALIC:
      fprintf (f, "{\\it ");
      break;
    case ST_SMALL_CAPS:
      fprintf (f, "{\\sc ");
      break;
    case ST_BOLD:
      fprintf (f, "{\\bf ");
      break;
    case ST_FORCED_CENTER:
    case ST_CENTER:
      fprintf (f, "}{");
      break;
    case ST_TT:
      fprintf (f, "{\\tt ");
      break;
    default:
      break;
    }
}

void
otex_write_end (FILE * f, unsigned char style)
{
  switch (style)
    {
    case ST_FORCED_CENTER:
    case ST_CENTER:
      fprintf (f, "}{");
      break;
    default:
      fprintf (f, "}");
      break;
    }
}

void
otex_write_special_char (FILE * f, grewchar * special_char)
{
  if (!gregorio_wcsbufcmp(special_char, "'æ"))
    {
      fprintf (f, "\\'ae");
      return;
    }
  if (!gregorio_wcsbufcmp(special_char, "'œ"))
    {
      fprintf (f, "\\'oe");
      return;
    }
  if (!gregorio_wcsbufcmp(special_char, "ae"))
    {
      fprintf (f, "\\ae");
      return;
    }
  if (!gregorio_wcsbufcmp(special_char, "R/"))
    {
      fprintf (f, "\\s R");
      return;
    }
  if (!gregorio_wcsbufcmp(special_char, "A/"))
    {
      fprintf (f, "\\s A");
      return;
    }
  if (!gregorio_wcsbufcmp(special_char, "V/"))
    {
      fprintf (f, "\\s V");
      return;
    }
}

void
otex_write_verb (FILE * f, grewchar * verb_str)
{
  gregorio_print_unistring  (f, verb_str);
}

void
otex_print_char (FILE * f, grewchar to_print)
{
  switch (to_print)
    {
    case L'œ':
      fprintf (f, "\\oe ");
      break;
    case L'æ':
      fprintf (f, "\\ae ");
      break;
    case L'é':
      fprintf (f, "\\'e ");
      break;
    case L'è':
      fprintf (f, "\\`e ");
      break;
    case L'à':
      fprintf (f, "\\`a ");
      break;
    case L'ô':
      fprintf (f, "\\^o ");
      break;
    case L'î':
      fprintf (f, "\\^i ");
      break;
    case L'í':
      fprintf (f, "\\'i ");
      break;
    case L'û':
      fprintf (f, "\\^u ");
      break;
    case L'ê':
      fprintf (f, "\\^e ");
      break;
    case L'ó':
      fprintf (f, "\\'o ");
      break;
    default:
      gregorio_write_one_tex_char(f, to_print);
      break;
    }
}

void
opustex_write_text (FILE * f, gregorio_character * text,
				char *first_syllable)
{
  if (text == NULL)
    {
      fprintf (f, "{}{}{}");
      return;
    }
  fprintf (f, "{");
  gregorio_write_text (*first_syllable, text, f,
			  (&otex_write_verb),
			  (&otex_print_char),
			  (&otex_write_begin),
			  (&otex_write_end),
			  (&otex_write_special_char));
  if (*first_syllable)
    {
      *first_syllable = 0;
    }
  fprintf (f, "}");
}


void
opustex_write_element (FILE * f, gregorio_element * element)
{
  gregorio_glyph *current_glyph = element->first_glyph;
  while (current_glyph)
    {
      if (current_glyph->type == GRE_SPACE)
	{
	  // we assume here that it is a SP_ZERO_WIDTH, the only one a glyph can be
	  if (loff != 1)
	    {
	      fprintf (f, "\\Nonspatium");
	    }
	  current_glyph = current_glyph->next;
	  continue;
	}
      if (current_glyph->type == GRE_FLAT)
	{
	  fprintf (f, "\\bmolle ");
	  opustex_print_note (f, current_glyph->glyph_type);
	  current_glyph = current_glyph->next;
	  continue;
	}
      if (current_glyph->type == GRE_NATURAL)
	{
	  fprintf (f, "\\bdurum ");
	  opustex_print_note (f, current_glyph->glyph_type);
	  current_glyph = current_glyph->next;
	  continue;
	}
      if (current_glyph->type == GRE_BAR)
	{
	  fprintf (f, "\\");
	  opustex_write_barline (f, current_glyph->glyph_type);
	  fprintf (f, "\n\\spatium\n");
	  current_glyph = current_glyph->next;
	  continue;
	}
      // at this point glyph->type is GRE_GLYPH
      opustex_write_glyph (f, current_glyph);
      current_glyph = current_glyph->next;
    }
}

void
opustex_write_barline (FILE * f, char type)
{
  switch (type)
    {
    case B_VIRGULA:
      fprintf (f, "virgula");
      break;
    case B_DIVISIO_MINIMA:
      fprintf (f, "divisiominima");
      break;
    case B_DIVISIO_MINOR:
      fprintf (f, "divisiominor");
      break;
    case B_DIVISIO_MAIOR:
      fprintf (f, "divisiomaior");
      break;
    case B_DIVISIO_FINALIS:
      fprintf (f, "divisiofinalis");
      break;
    default:
      gregorio_message (_("unknown bar type"),
			   "opustex_write_barline", ERROR, 0);
      break;
    }
}

void
opustex_write_finis (FILE * f, char type)
{
  switch (type)
    {
    case B_DIVISIO_MAIOR:
      fprintf (f, "finisgregoriana");
      break;
    default:
      fprintf (f, "Finisgregoriana");
      break;
    }
}

void
opustex_write_glyph (FILE * f, gregorio_glyph * glyph)
{
  gregorio_note *current_note = NULL;
  int h_length = 0;
  int h_pitch = 0;
  unsigned char augmentum;

  if (!glyph)
    {
      gregorio_message (_
			   ("called with NULL pointer"),
			   "opustex_write_glyph", ERROR, 0);
      return;
    }
  if (glyph->type == GRE_TEXVERB_GLYPH)
    {
      if (glyph->texverb)
        {
          fprintf(f, "%s", glyph->texverb);
        }
      return;
    }
  if (!glyph->first_note)
    {
      gregorio_message (_
			   ("called with glyph without note"),
			   "opustex_write_glyph", ERROR, 0);
      return;
    }
  if (loff == 1)
    {
      fprintf (f, "\\loff{");
      loff++;
    }
  current_note = glyph->first_note;
  while (current_note)
    {
      if (current_note->signs >= _V_EPISEMUS)
	{
	  fprintf (f, "\\ictus ");
	  opustex_print_note (f, current_note->pitch);
	}
      current_note = current_note->next;
    }
  current_note = glyph->first_note;
  // divide puncta_inclinata into single glyphs
  if (is_puncta_inclinata (glyph->glyph_type)
      || glyph->glyph_type == G_PUNCTA_INCLINATA)
    {
      if (glyph->first_note->signs >= _V_EPISEMUS)
	{
	  fprintf (f, "\\ictus ");
	  opustex_print_note (f, glyph->first_note->pitch);
	}
      fprintf (f, "\\punctuminclinatum ");
      opustex_print_note (f, glyph->first_note->pitch);
      current_note = glyph->first_note->next;
      while (current_note)
	{
	  fprintf (f, "\\nonspatium");
	  if (simple_htype(current_note->h_episemus_type == H_ALONE))
	    {
	      opustex_print_episem (f, current_note->pitch, 1);
	    }
	  if (current_note->signs >= _V_EPISEMUS)
	    {
	      fprintf (f, "\\ictus ");
	      opustex_print_note (f, current_note->pitch);
	    }
	  fprintf (f, "\\punctuminclinatum ");
	  opustex_print_note (f, current_note->pitch);
	  current_note = current_note->next;
	}
      return;
    }
  else
    {
      if (glyph->glyph_type != G_PES)
	{
	  while (current_note)
	    {
	      if (simple_htype(current_note->h_episemus_type == H_ALONE))
		{
		  opustex_print_episem (f, current_note->pitch,
						    1);
		}
	      if (simple_htype(current_note->h_episemus_type == H_MULTI))
		{
		  h_length++;
		  h_pitch = current_note->h_episemus_top_note;
		}
	      current_note = current_note->next;
	    }
	  if (h_length > 0)
	    {
	      opustex_print_episem (f, h_pitch, h_length);
	    }
	}
      else
	{
	  if (simple_htype(glyph->first_note->h_episemus_type) != H_NO_EPISEMUS)
	    {
	      opustex_print_episem_under (f,
						      glyph->first_note->
						      pitch, 1);
	    }
	  if (simple_htype(glyph->first_note->next->h_episemus_type) != H_NO_EPISEMUS)
	    {
	      opustex_print_episem (f,
						glyph->first_note->next->
						pitch, 1);
	    }
	}

    }
  // special shapes
  if (glyph->glyph_type == G_DISTROPHA_AUCTA)
    {
      fprintf (f, "\\strophaaucta ");
      opustex_print_note (f, glyph->first_note->pitch);
      fprintf (f, "\\spatiumparvum\\strophaaucta ");
      opustex_print_note (f, glyph->first_note->pitch);
      return;
    }
  if (glyph->glyph_type == G_TRISTROPHA_AUCTA)
    {
      fprintf (f, "\\strophaaucta ");
      opustex_print_note (f, glyph->first_note->pitch);
      fprintf (f, "\\spatiumparvum\\strophaaucta ");
      opustex_print_note (f, glyph->first_note->pitch);
      fprintf (f, "\\spatiumparvum\\strophaaucta ");
      opustex_print_note (f, glyph->first_note->pitch);
      return;
    }
  if (glyph->glyph_type == G_PUNCTUM && glyph->first_note->shape == S_ORISCUS)
    {
      fprintf (f, "\\oriscus ");
      opustex_print_note (f, glyph->first_note->pitch);
      return;
    }
  if (glyph->glyph_type == G_PUNCTUM
      && glyph->first_note->shape == S_ORISCUS_AUCTUS)
    {
      fprintf (f, "\\oriscusreversus ");
      opustex_print_note (f, glyph->first_note->pitch);
      return;
    }
  if (glyph->glyph_type == G_PUNCTUM
      && glyph->first_note->shape == S_QUILISMA)
    {
      fprintf (f, "\\quilisma ");
      opustex_print_note (f, glyph->first_note->pitch);
      return;
    }
  // pes quassus
  if (glyph->glyph_type == G_PES && glyph->first_note->shape == S_ORISCUS)
    {
      fprintf (f, "\\pesquassus ");
    }
  else
    {
      // quilisma
      if (glyph->first_note->shape == S_QUILISMA
	  && (glyph->glyph_type == G_PES || glyph->glyph_type == G_TORCULUS))
	{
	  fprintf (f, "\\%s%s", "quilisma",
		   opustex_glyph_type_to_str (glyph->glyph_type));
	}
      else
	{
	  if (glyph->first_note->shape == S_QUILISMA
	      && glyph->glyph_type == G_PES_QUADRATUM)
	    {
	      fprintf (f, "\\%s%s", "Quilisma",
		       opustex_glyph_type_to_str (glyph->
							      glyph_type));
	    }
	}
    }

  if (glyph->liquescentia)
    {
      opustex_print_liquescentia (f, glyph->liquescentia,
					      glyph->glyph_type);
    }
  fprintf (f, " ");
  //print the notes of the glyph
  current_note = glyph->first_note;
  while (current_note)
    {
      opustex_print_note (f, current_note->pitch);
      current_note = current_note->next;
    }

  current_note = glyph->first_note;
  while (current_note)
    {
      if (current_note->signs == _PUNCTUM_MORA
	  || current_note->signs == _V_EPISEMUS_PUNCTUM_MORA)
	{
	  fprintf (f, "\\augmentum ");
	  opustex_print_augmentum_note (f, current_note->pitch);
	  augmentum = 1;
	}
      if (current_note->signs == _AUCTUM_DUPLEX
	  || current_note->signs == _V_EPISEMUS_AUCTUM_DUPLEX)
	{
	  fprintf (f, "\\augmentumduplex ");
	  opustex_print_augmentum_note (f, current_note->pitch);
	  opustex_print_augmentum_note (f,
						    current_note->
						    previous->pitch);
	  augmentum = 1;
	}
      current_note = current_note->next;
    }

  if (augmentum == 1 && glyph->next
      && glyph->next->type == GRE_SPACE
      && glyph->next->glyph_type == SP_ZERO_WIDTH)
    loff++;
}

void
opustex_print_note (FILE * f, char pitch)
{
  if (is_even (otexclef))
    {
      if (pitch - otexclef < 104)
	{
	  fprintf (f, "%c", pitch - otexclef - 25);
	}
      else
	{
	  fprintf (f, "%c", pitch - otexclef - 7);
	}
    }
  else
    {
      if (pitch - otexclef < 97)
	{
	  fprintf (f, "%c", pitch - otexclef - 18);
	}
      else
	{
	  fprintf (f, "%c", pitch - otexclef);
	}
    }
}

void
opustex_print_episem (FILE * f, char pitch, char length)
{
  int realpitch;

  if (is_even (otexclef))
    {
      if (pitch - otexclef < 104)
	{
	  realpitch = pitch - otexclef - 25;
	}
      else
	{
	  realpitch = pitch - otexclef - 7;
	}
    }
  else
    {
      if (pitch - otexclef < 97)
	{
	  realpitch = pitch - otexclef - 18;
	}
      else
	{
	  realpitch = pitch - otexclef;
	}
    }
  if (!is_even (pitch) && pitch < 'k')	// if the note is between staff lines
    {
      fprintf (f, "\\episem %c%d", realpitch + 2, length);
    }
  else
    {
      fprintf (f, "\\episem %c%d", realpitch + 1, length);
    }
}

void
opustex_print_episem_under (FILE * f, char pitch, char length)
{
  int realpitch;

  if (is_even (otexclef))
    {
      if (pitch - otexclef < 104)
	{
	  realpitch = pitch - otexclef - 25;
	}
      else
	{
	  realpitch = pitch - otexclef - 7;
	}
    }
  else
    {
      if (pitch - otexclef < 97)
	{
	  realpitch = pitch - otexclef - 18;
	}
      else
	{
	  realpitch = pitch - otexclef;
	}
    }
  if (!is_even (pitch) && pitch > 'c')	// if the note is between staff lines
    {
      fprintf (f, "\\episem %c%d", realpitch - 2, length);
    }
  else
    {
      fprintf (f, "\\episem %c%d", realpitch - 1, length);
    }
}


void
opustex_print_augmentum_note (FILE * f, char pitch)
{
  int realpitch;

  if (is_even (otexclef))
    {
      if (pitch - otexclef < 104)
	{
	  realpitch = pitch - otexclef - 25;
	}
      else
	{
	  realpitch = pitch - otexclef - 7;
	}
    }
  else
    {
      if (pitch - otexclef < 97)
	{
	  realpitch = pitch - otexclef - 18;
	}
      else
	{
	  realpitch = pitch - otexclef;
	}
    }
  if (is_even (realpitch))
    {
      fprintf (f, "%c", realpitch);
    }
  else
    {
      fprintf (f, "%c", realpitch + 1);
    }
}

const char *
opustex_glyph_type_to_str (char name)
{
  const char *str = "";
  switch (name)
    {
    case G_PUNCTUM_INCLINATUM:
      str = "punctuminclinatum";
      break;
    case G_TRIGONUS:
      str = "trigonus";
      break;
    case G_VIRGA:
      str = "virga";
      break;
    case G_STROPHA:
      str = "stropha";
      break;
    case G_PUNCTUM:
      str = "punctum";
      break;
    case G_PES:
      str = "pes";
      break;
    case G_PES_QUADRATUM:	//doesn't exist in OpusTeX
      str = "pes";
      gregorio_message (_("pes quadratum doesn't exist in OpusTeX"),
			   "opustex_glyph_type_to_str", ERROR, 0);
      break;
    case G_FLEXA:
      str = "clivis";
      break;
    case G_TORCULUS:
      str = "torculus";
      break;
    case G_TORCULUS_RESUPINUS:
      str = "torculusresupinus";
      break;
    case G_TORCULUS_RESUPINUS_FLEXUS:
      gregorio_message (_
			   ("torculus_resupinus_flexus doesn't exist in OpusTeX"),
			   "opustex_glyph_type_to_str", ERROR, 0);
      break;
    case G_PORRECTUS:
      str = "porrectus";
      break;
    case G_PORRECTUS_FLEXUS:
      str = "porrectusflexus";
      break;
    case G_BIVIRGA:
      str = "varbivirga";
      break;
    case G_TRIVIRGA:
      str = "vartrivirga";
      break;
    case G_DISTROPHA:
      str = "distropha";
      break;
    case G_TRISTROPHA:
      str = "tristropha";
      break;
    case G_SCANDICUS:
      str = "scandicus";
      break;
    default:
      break;
    }
  return str;
}

void
opustex_print_liquescentia (FILE * f, char liquescentia,
					char glyph)
{
  const char *suffix = "us";
  if (glyph == G_FLEXA || glyph == G_STROPHA)
    {
      suffix = "a";
    }
  if (glyph <= G_PUNCTUM_INCLINATUM)
    {
      suffix = "um";
    }
  switch (liquescentia)
    {
    case L_DEMINUTUS:
      if (glyph == G_TORCULUS || glyph == G_PORRECTUS
	  || glyph == G_TORCULUS_RESUPINUS)
	{
	  fprintf (f, "deminutus");
	}
      else
	{
	  gregorio_message (_("that glyph cannot be deminutus in OpusTeX"),
			       "opustex_print_liquescentia",
			       ERROR, 0);
	}
      break;
    case L_AUCTUS_ASCENDENS:
      if (glyph == G_PUNCTUM || glyph == G_PES || glyph == G_FLEXA)
	{
	  fprintf (f, "auct%sascendens", suffix);
	}
      else
	{
	  gregorio_message (_
			       ("that glyph cannot be auctus ascendens in OpusTeX"),
			       "opustex_print_liquescentia",
			       ERROR, 0);
	}
      break;
    case L_AUCTUS_DESCENDENS:
      if (glyph == G_PUNCTUM || glyph == G_PES || glyph == G_FLEXA
	  || glyph == G_TORCULUS || glyph == G_PORRECTUS)
	{
	  fprintf (f, "auct%sdescendens", suffix);
	}
      else
	{
	  gregorio_message (_
			       ("that glyph cannot be auctus descendens in OpusTeX"),
			       "opustex_print_liquescentia",
			       ERROR, 0);
	}
      break;
    case L_AUCTA:
      if (glyph == G_PUNCTUM_INCLINATUM || glyph == G_STROPHA)
	{
	  fprintf (f, "auct%s", suffix);
	}
      else
	{
	  gregorio_message (_("that glyph cannot be auctus in OpusTeX"),
			       "opustex_print_liquescentia",
			       ERROR, 0);
	}
      break;
    case L_INITIO_DEBILIS:
      if (glyph == G_PES || glyph == G_TORCULUS || glyph == G_PORRECTUS)
	{
	  fprintf (f, "initiodebilis");
	}
      else
	{
	  gregorio_message (_
			       ("that glyph cannot have initio debilis in OpusTeX"),
			       "opustex_print_liquescentia",
			       ERROR, 0);
	}
      break;
    case L_DEMINUTUS_INITIO_DEBILIS:
      if (glyph == G_TORCULUS)
	{
	  fprintf (f, "deminitusinitiodebilis");
	}
      else
	{
	  gregorio_message (_
			       ("that glyph cannot be deminutus initio debilis in OpusTeX"),
			       "opustex_print_liquescentia",
			       ERROR, 0);
	}
      break;
    case L_AUCTUS_ASCENDENS_INITIO_DEBILIS:
      gregorio_message (_
			   ("there's no auctus ascendens initio debilis in OpusTeX"),
			   "opustex_print_liquescentia", ERROR,
			   0);
      break;
    case L_AUCTUS_DESCENDENS_INITIO_DEBILIS:
      if (glyph == G_PES || glyph == G_TORCULUS)
	{
	  fprintf (f, "auctus descendens initiodebilis");
	}
      else
	{
	  gregorio_message (_
			       ("that glyph cannot be auctus descendens initio debilis in OpusTeX"),
			       "opustex_print_liquescentia",
			       ERROR, 0);
	}
      break;
    case L_AUCTA_INITIO_DEBILIS:
      gregorio_message (_("there's no aucta initio debilis in OpusTeX"),
			   "opustex_print_liquescentia", ERROR,
			   0);
      break;
    }
}

char
find_next_note (gregorio_element * current_element,
			    gregorio_syllable * current_syllable)
{
  gregorio_element *next_element = current_element->next;
  gregorio_element *element_element = NULL;
  char stop = 0;
  gregorio_glyph *next_glyph = NULL;

  while (stop == 0)
    {
      if (next_element)
	{
	  if (next_element->type == GRE_ELEMENT)
	    {
	      element_element = next_element;
	      stop = 1;
	    }
	  else
	    {
	      next_element = next_element->next;
	    }
	}
      else
	{
	  if (current_syllable->next_syllable)
	    {
	      current_syllable = current_syllable->next_syllable;
	      next_element = current_syllable->elements[0];
	    }
	  else
	    {
	      stop = 1;
	    }
	}
    }
  if (element_element)
    {
      next_glyph = element_element->first_glyph;
      while (next_glyph->type != GRE_GLYPH)
	{
	  next_glyph = next_glyph->next;
	}
    }
  if (next_glyph)
    {
      return next_glyph->first_note->pitch;
    }
  return 0;
}
