/*
 *  @(#)IEditableIssue.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.pmti.v1;

 

/**
 * Allows for editing of an issue.  The only parts that can't be edited are
 * the ID and type, since those uniquely identify the issue at hand.  Editing
 * an issue has several constraints that should be known by the user:
 *  <UL>
 *      <LI>
 *      Just like with the <tt>IIssue</tt> instances, the issue being
 *      edited will NOT be real-time updated to reflect the current
 *      tracker state.  Currently, the only way to update an issue is by
 *      re-polling the <tt>ProblemManager</tt>.  Individual implementations
 *      may provide for alternative means to receive synchronized issues.
 *      </LI>
 *      <LI>
 *      No changes to an editable issue will be committed to the problem
 *      tracker is to call <tt>commit()</tt> on the issue.
 *      </LI>
 *  </UL>
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:51:54 $
 * @since      July 6, 2002
 */
public interface IEditableIssue extends IIssue
{
    
    
    /**
     * 
     */
    public void setShortDescription( String desc );
    
    
    /**
     * @return <tt>true</tt> if <tt>setShortDescription( String )</tt> was
     *      called with a different description string than the original
     *      issue, otherwise <tt>false</tt>.
     */
    public boolean hasShortDescriptionChanged();
    
    
    /**
     * Returns the list of all states that this issue can move to next.
     * This is part of the workflow logic of the underlying PMT.  The returned
     * states may be safely edited without any affect; the only effect will be
     * when the state is explicitly set.  This will always return, in index 0,
     * a <b>copy</b> of the current state as editable.
     */
    public IEditableIssueState[] getNextStates();
    
    
    /**
     * Sets the current state.  Since there is no getEditableState() method,
     * use this method if any information in the current state needs to be
     * updated.  You can retrieve the current state as an editable state
     * using <tt>getNextStates()[0]</tt>, but note that any changes to that
     * editable version will not affect the tracker's state unless that
     * editable instance is explicitly set in this method.
     *
     * @exception ProblemManagerException if the input state is not a valid
     *      next state.
     */
    public void setState( IIssueState state )
            throws ProblemManagerException;
    
    
    /**
     * @return <tt>true</tt> if the <tt>setState( IIssueState )</tt> method
     *      has been invoked and did not throw an exception, otherwise
     *      <tt>false</tt>.  Note that even if the set state is an unchanged
     *      version of the current issue's state, this will still return
     *      <tt>true</tt>.
     */
    public boolean hasStateChanged();
    
    
    /**
     * This is a synonymn for <tt>getAttributes()</tt>, but this explicitly
     * sets the returned value as an editable set, without the need for an
     * extra cast.  The returned attribute set may be safely edited, and
     * changes there will affect the issue that returned them.
     */
    public IEditableAttributeSet getEditableAttributes();
    
    
    /**
     * Commits all changes from the issue to the tracker.
     * <P>
     * In theory, issues should never be removed.  However, some systems allow
     * them to be deleted (say, if there was an accidental creation).  In this
     * case, an <tt>IssueRemovedException</tt> will be thrown.
     *
     * @exception ProblemManagerException if there was an underlying tracker
     *      error.
     */
    public void commit()
            throws ProblemManagerException;
}

