/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.interceptors;

import org.jboss.cache.CacheException;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.NodeSPI;
import org.jboss.cache.commands.VisitableCommand;
import org.jboss.cache.commands.tx.CommitCommand;
import org.jboss.cache.commands.tx.OptimisticPrepareCommand;
import org.jboss.cache.commands.tx.RollbackCommand;
import org.jboss.cache.factories.annotations.Start;
import static org.jboss.cache.lock.LockType.READ;
import static org.jboss.cache.lock.LockType.WRITE;
import org.jboss.cache.optimistic.TransactionWorkspace;
import org.jboss.cache.optimistic.WorkspaceNode;
import org.jboss.cache.transaction.GlobalTransaction;
import org.jboss.cache.transaction.TransactionEntry;

/**
 * Locks nodes during transaction boundaries.  Only affects prepare/commit/rollback method calls; other method calls
 * are simply passed up the interceptor stack.
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 * @author <a href="mailto:stevew@jofti.com">Steve Woodcock (stevew@jofti.com)</a>
 */
public class OptimisticLockingInterceptor extends OptimisticInterceptor
{
   @Start
   private void init()
   {
      if (txManager == null)
         log.fatal("No transaction manager lookup class has been defined. Transactions cannot be used and thus OPTIMISTIC locking cannot be used!  Expect errors!!");
   }

   @Override
   public Object visitOptimisticPrepareCommand(InvocationContext ctx, OptimisticPrepareCommand command) throws Throwable
   {
      //try and acquire the locks - before passing on
      GlobalTransaction gtx = getGlobalTransaction(ctx);

      boolean succeeded = false;
      try
      {
         TransactionWorkspace<?, ?> workspace = getTransactionWorkspace(ctx);
         if (log.isDebugEnabled()) log.debug("Locking nodes in transaction workspace for GlobalTransaction " + gtx);

         for (WorkspaceNode workspaceNode : workspace.getNodes().values())
         {
            NodeSPI node = workspaceNode.getNode();

            boolean isWriteLockNeeded = workspaceNode.isDirty() || (workspaceNode.isChildrenModified() && (configuration.isLockParentForChildInsertRemove() || node.isLockForChildInsertRemove()));

            boolean acquired = lockManager.lockAndRecord(node, isWriteLockNeeded ? WRITE : READ, ctx);
            if (acquired)
            {
               if (trace) log.trace("Acquired lock on node " + node.getFqn());
            }
            else
            {
               throw new CacheException("Unable to acquire lock on node " + node.getFqn());
            }

         }

         // locks have acquired so lets pass on up
         Object retval = invokeNextInterceptor(ctx, command);
         succeeded = true;
         return retval;
      }
      catch (Throwable e)
      {
         succeeded = false;
         log.debug("Caught exception attempting to lock nodes ", e);
         //we have failed - set to rollback and throw exception
         throw e;
      }
      finally
      {
         if (!succeeded) unlock(ctx);
      }
   }

   @Override
   public Object visitCommitCommand(InvocationContext ctx, CommitCommand command) throws Throwable
   {
      return transactionFinalized(ctx, command);
   }

   @Override
   public Object visitRollbackCommand(InvocationContext ctx, RollbackCommand command) throws Throwable
   {
      return transactionFinalized(ctx, command);
   }

   private Object transactionFinalized(InvocationContext ctx, VisitableCommand command) throws Throwable
   {
      Object retval = null;
      // we need to let the stack run its commits or rollbacks first -
      // we unlock last - even if an exception occurs
      try
      {
         retval = invokeNextInterceptor(ctx, command);
      }
      finally
      {
         unlock(ctx);
      }
      return retval;
   }

   /**
    * Releases all locks held by the specified global transaction.
    *
    * @param ctx Invocation Context
    */
   private void unlock(InvocationContext ctx)
   {
      try
      {
         TransactionEntry entry = ctx.getTransactionEntry();
         if (entry != null)
         {
            lockManager.unlock(ctx);
         }
      }
      catch (Exception e)
      {
         // we have failed to unlock - now what?
         log.error("Failed to unlock nodes after a commit or rollback!  Locks are possibly in a very inconsistent state now!", e);
      }
   }

}
