////////////////////////////////////////////////////////////////////////////////
//
//	Pico Technology USB TC08 Driver
//
///	\file     TC08DeviceInternalData.cpp
///	\brief    Internals of the TC08 device
//
//	Copyright (c) 2007, Pico Technology.
//	All rights reserved.
//   
//	Redistribution and use in source and binary forms, with or without
//	modification, are permitted provided that the following conditions are met:
//		* Redistributions of source code must retain the above copyright
//		  notice, this list of conditions and the following disclaimer.
//		* Redistributions in binary form must reproduce the above copyright
//		  notice, this list of conditions and the following disclaimer in the
//		  documentation and/or other materials provided with the distribution.
//		* The name of Pico Technology may not be used to endorse or promote
//		  products derived from this software without specific prior written
//		  permission.
//
//	THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY ``AS IS'' AND ANY
//	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
//	DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//	ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//	THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//	Version $Id: TC08DeviceInternalData.cpp,v 1.8 2007/08/02 09:48:12 douglas Exp $
//
////////////////////////////////////////////////////////////////////////////////


#include <assert.h>
#include <stdio.h>
#include <ctype.h>
#include <limits>
#include <math.h>
#include <pthread.h>
#include <unistd.h>

#include <sys/times.h>

#include "../shared/PicoStatus.h"
#include "../shared/PicoPortability.h"
#include "../shared/PicoUsbDevice.h"
#include "TC08Device.h"
#include "TC08DeviceInternalData.h"
#include "readingbuff.h"
#include "tctables.h"

#include "lookupdata.h"
#include "lookupdatatable.h"
	

////////////////////////////////////////////////////////////////////////////////
// STATIC VARIABLES
////////////////////////////////////////////////////////////////////////////////

/// Flag to indicate whether the static thermocouple tables have already been 
/// initialized. Should be set by the first instance of the class to be
/// created.
int TC08DeviceInternalData::tablesInitialized = 0;


ThermocoupleTables TC08DeviceInternalData::TCTables;

//////////////////////////////////////////////////////////////////////////
/// Set each channel's circ buffer to empty
//////////////////////////////////////////////////////////////////////////
void TC08DeviceInternalData::ResetBuffers(void) {
	for (int curCh=0;curCh<USBTC08_CHANNEL_COUNT;curCh++) {
		Channels[curCh].SampleBuffer.ResetBuffers();
	}
}


//////////////////////////////////////////////////////////////////////////
/// Constructor - initializes the thermocouple lookup tables if this hasn't 
/// already been done.
//////////////////////////////////////////////////////////////////////////
TC08DeviceInternalData::TC08DeviceInternalData(void)
{
	if(!tablesInitialized) {
		tablesInitialized=true;
		InitializeTables();
	}
	threadStop = false;
	pthread_mutex_init(&threadLock, NULL);
	info=NULL;
	freq = USBTC08_MAINS_50HZ;
	for(int i=0;i<USBTC08_CHANNEL_COUNT;i++) {
		LastGoodValues[i]=NAN;
	}

}


//////////////////////////////////////////////////////////////////////////
/// Set up thermocouple lookup tables		
//////////////////////////////////////////////////////////////////////////
void TC08DeviceInternalData::InitializeTables(void) {
#if DEBUG
	printf("TC08InternalDevice::InitializeTables()\n");
#endif
	
	/* **** TODO **** */
	// Add locking mechanism here
	
	// Set up thermocouple lookup tables		
	int             ThermocoupleTypeCount;
	int             ThermocoupleType;
	USBTC08_TCTABLE ThermocoupleTypes[] = 
	{
	{'C', LIMITS_THERMISTOR_MIN, LIMITS_THERMISTOR_MAX, LIMITS_THERMISTOR_INT},
	{'B', LIMITS_TYPEB_MIN, LIMITS_TYPEB_MAX, LIMITS_TYPEB_INT},
	{'E', LIMITS_TYPEE_MIN, LIMITS_TYPEE_MAX, LIMITS_TYPEE_INT},
	{'J', LIMITS_TYPEJ_MIN, LIMITS_TYPEJ_MAX, LIMITS_TYPEJ_INT},
	{'K', LIMITS_TYPEK_MIN, LIMITS_TYPEK_MAX, LIMITS_TYPEK_INT},
	{'N', LIMITS_TYPEN_MIN, LIMITS_TYPEN_MAX, LIMITS_TYPEN_INT},
	{'R', LIMITS_TYPER_MIN, LIMITS_TYPER_MAX, LIMITS_TYPER_INT},
	{'S', LIMITS_TYPES_MIN, LIMITS_TYPES_MAX, LIMITS_TYPES_INT},
	{'T', LIMITS_TYPET_MIN, LIMITS_TYPET_MAX, LIMITS_TYPET_INT}
	};
	
	ThermocoupleTypeCount = (int)(sizeof(ThermocoupleTypes) / sizeof(ThermocoupleTypes[0]));
	
	for (ThermocoupleType = 0; ThermocoupleType < ThermocoupleTypeCount; ++ThermocoupleType) {		
		// store a reference to this data in the TCTable class
		TCTables.AddTable(  ThermocoupleTypes[ThermocoupleType].TCType, 
							ThermocoupleTable[ThermocoupleType],
							ThermocoupleTypes[ThermocoupleType].MinTemp,
							ThermocoupleTypes[ThermocoupleType].MaxTemp,
							ThermocoupleTypes[ThermocoupleType].Interval);
	}
}

//////////////////////////////////////////////////////////////////////////
/// Read bytes from the TC08 unit over USB
/// Note that the TC08Device does not need to be open to do this but that
/// the PicoUsbDevice must be.
/// <param name="buf">Pointer to a location to store the read bytes</param>
/// <param name="size">Number of bytes to attempt to read</param>
/// <returns>The return value of PicoUsbDevice::ReadPipe</returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08DeviceInternalData::ReadPipe(unsigned char *buf,unsigned int *size) {
	// Ensure we have a valid device
 	assert(usbDevice);

	return usbDevice->ReadPipe(USBTC08_BULKIN_PIPE_ID, buf, size);
}
//////////////////////////////////////////////////////////////////////////
/// Write bytes to the TC08 unit over USB
/// Note that the TC08Device does not need to be open to do this but that
/// the PicoUsbDevice must be.
/// <param name="buf">Pointer to the bytes to be written</param>
/// <param name="size">Number of bytes to attempt to write</param>
/// <returns>The return value of PicoUsbDevice::WritePipe</returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08DeviceInternalData::WritePipe(unsigned char *buf,unsigned int size) {

	assert(usbDevice);

	return usbDevice->WritePipe(USBTC08_BULKOUT_PIPE_ID,buf,size);
}

//////////////////////////////////////////////////////////////////////////
/// Flush the USB in and out pipes
/// Note that the TC08Device does not need to be open to do this but that
/// the PicoUsbDevice must be.
/// <returns>The return value of PicoUsbDevice::ResetPipe</returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08DeviceInternalData::ResetPipes(void) {

 	assert(usbDevice);

 	PICO_RETURNS err = usbDevice->ResetPipe(USBTC08_BULKOUT_PIPE_ID);
	
	if(err==PICO_SUCCESS) {
	
		err = usbDevice->ResetPipe(USBTC08_BULKIN_PIPE_ID);
		
	}
	
	return err;
}

//////////////////////////////////////////////////////////////////////////
/// Send the Init command to the device 
/// Must be done after opening before any other commands
//////////////////////////////////////////////////////////////////////////
void TC08DeviceInternalData::Init(void) {
	
	unsigned char buf[USBTC08_CMD_BUF_SIZE];

#if DEBUG
	printf("TC08Device::Init()\n");
#endif
	
	assert(usbDevice);
#if DEBUG
	printf("TC08Device::Init(), device OK\n");
#endif
	assert(usbDevice->GetDeviceState()==PICODEVICE_STATE_OPEN);
#if DEBUG
	printf("TC08Device::Init(), state OK\n");
#endif
	
	buf[0]=USBTC08_CMD_INIT;
#if DEBUG
	printf("TC08Device::Init(), about to write to device\n");
#endif
	WritePipe(buf,1);
}

//////////////////////////////////////////////////////////////////////////
/// Private function to write the channel config to the unit
//////////////////////////////////////////////////////////////////////////
bool TC08DeviceInternalData::SetAllChannels ()
{
	int   iChannelIndex;
	char  bChannelCount;
	unsigned char buf[USBTC08_CMD_BUF_SIZE];
	
  // Check the device is still open (calling fn should have checked if it is streaming)
	assert(usbDevice);
	if(usbDevice->GetDeviceState()!=PICODEVICE_STATE_OPEN) {
    return false;
  }
	
	memset(buf, 0, sizeof(buf));
	
	// TODO: Windows driver resets USB buffers here. Is that necessary?
	
  // Start with the CJC disabled
	Channels[USBTC08_CHANNEL_CJC].fEnabled=false;
	Channels[USBTC08_CHANNEL_CJC].bTCType=' ';
	
  // Setup the thermocouple channels (1-8)
	for(iChannelIndex = 1, bChannelCount = 0; iChannelIndex <= USBTC08_MAX_CHANNELS; ++iChannelIndex) {
		
		if(Channels[iChannelIndex].fEnabled) {
			buf[(iChannelIndex * 2) + 2] = 1;
			
			++bChannelCount;
			
			// if we are converting at least one thermocouple then we need the CJC
			if (Channels[iChannelIndex].bTCType != 'X') {
				Channels[USBTC08_CHANNEL_CJC].fEnabled=true;
				Channels[USBTC08_CHANNEL_CJC].bTCType='C';
			}
			
			buf[(iChannelIndex * 2) + 3] = (USBTC08_GAIN_THERMOCOUPLE / (UInt8)USBTC08_FIXED_AMP_GAIN);
		} else {
			buf[(iChannelIndex * 2) + 2] = 0;
		}
    }
	
	
	
	buf[0] = USBTC08_CMD_SETUP_CHANNELS;
	
	
	// Now setup the CJC if it's required
	if(CJCUsed()) {
		buf[1]=bChannelCount+1;
		buf[2]=1;
		buf[3]=USBTC08_GAIN_1V25; // CJC Gain (1.25V)
		bChannelCount++;
    } else {
		buf[1]=bChannelCount;
		buf[2]=0;
	}

	// Write the setup data to the unit
	WritePipe(buf,20);
	
	return true;
}

//////////////////////////////////////////////////////////////////////////
/// Limit overrange readings to the "maximum" allowable value
//////////////////////////////////////////////////////////////////////////
bool TC08DeviceInternalData::LimitReading(float *reading, float limit) {
	bool overflow;
	
	overflow = ((*reading > limit) || (*reading < -limit));
	
	*reading = __min(*reading, limit);
	*reading = __max(*reading, -limit);
	
	return !overflow;
}


//////////////////////////////////////////////////////////////////////////
/// Is the CJC used? True if either it is enabled itself, or any other
/// channel is set for thermocouple measurement
//////////////////////////////////////////////////////////////////////////
bool TC08DeviceInternalData::CJCUsed(void) {
	return (Channels[USBTC08_CHANNEL_CJC].fEnabled==true)&&(Channels[USBTC08_CHANNEL_CJC].bTCType=='C');
}



//////////////////////////////////////////////////////////////////////////
/// Function started as pthread for streaming
//////////////////////////////////////////////////////////////////////////
void *TC08DeviceInternalData::threadProc(void *args) {

#if DEBUG
	printf("TC08Device::threadProc(%p)\n",args);

	printf("Dev is %d\n", ((TC08DeviceInternalData *)args));
	
	printf("Interval is %d\n",((TC08DeviceInternalData *)args)->interval);
#endif
	
	if(args!=NULL) {
		((TC08DeviceInternalData *)args)->PollUnitThreaded();
	}
	return NULL;
}

//////////////////////////////////////////////////////////////////////////
/// Continuously collect readings from the unit in streaming mode and
/// write them to the driver's circular buffer
//////////////////////////////////////////////////////////////////////////
void TC08DeviceInternalData::PollUnitThreaded() {
	int             iChannelIndex;
	int             iReadings2Collect;
	int             iReading;
	long            lADCCount;
	float           flScaledValue;
	float           flCJCTemp;
	float           flCJCMilliVolts;
	float           flMilliVolts;
	bool            fConverted;
	bool            fOverflow;
	bool            tempOverflow;
	unsigned int	readCount;
	unsigned char buf[USBTC08_CMD_BUF_SIZE];
	
#if DEBUG
	printf("TC08Device::PollUnitThreaded()\n");
#endif
	
	int terminate = 0;
	clock_t PollStartTime = GetTickCount();
#if DEBUG
	printf("TC08Device::PollUnitThreaded: PollStartTime=%i\n",PollStartTime);
#endif	
	
	// Set the last sample time as the current time 
	//  so we don't take a reading straight away
	//  (the first reading is garbage & 
	//  the USB TC08 disposes of it internally)
	//
	clock_t LastSampleTime=PollStartTime;
#if DEBUG
	printf("TC08Device::PollUnitThreaded: LastSampleTime=%i\n",LastSampleTime);
#endif	
	
	do {
#if DEBUG
		//printf("TC08Device::PollUnitThreaded: Unlocking thread lock\n");
#endif
		//pthread_mutex_unlock(&threadLock);
#if DEBUG
		//printf("TC08Device::PollUnitThreaded: Unlocked thread lock\n");
#endif
		
		// Workout how many sample intervals have elapsed since the last read
		assert(interval);
		iReadings2Collect = (int)((GetTickCount()-LastSampleTime)/interval);
		#if DEBUG
			printf("TC08Device::PollUnitThreaded: iReadings2Collect=%i\n",iReadings2Collect);
		#endif
		
		// Note the time that we marked the number of readings to collect
		LastSampleTime = GetTickCount();
		#if DEBUG
			printf("TC08Device::PollUnitThreaded: LastSampleTime=%i\n",LastSampleTime);
		#endif
		
		
		// Only need to pad the sample buffer with QNaN floats if we are in
		//  'normal' streaming mode
		
		if (iReadings2Collect > 2) {
			for (iChannelIndex = 0; iChannelIndex <= USBTC08_MAX_CHANNELS; ++iChannelIndex) {
				if (Channels[iChannelIndex].fEnabled) {
					// Pad the missing readings with QNaN values (see IEEE 754)
					Channels[iChannelIndex].SampleBuffer.PadQNaN((long)iReadings2Collect - 2L);
				}
			}
		}
		
		
		if (iReadings2Collect > 2) {
			iReadings2Collect = 2;
		}
		
		//ErrorCode = USBTC08_ERROR_OK;
		for (iReading = 0; iReading < iReadings2Collect; ++iReading) {
			// If this fails we can pad the buffers
			//  and report QNaNs
			
#if DEBUG
			printf("TC08Device::PollUnitThreaded: Reading...");
#endif
			readCount=USBTC08_STREAM_PKT_SIZE;
			if(ReadPipe(buf,&readCount) != PICO_SUCCESS) {
#if DEBUG
				printf("error!\n");
#endif
				
				for (iChannelIndex = 0; iChannelIndex <= USBTC08_MAX_CHANNELS; ++iChannelIndex) {
					if (Channels[iChannelIndex].fEnabled) {
						// Pad the missing readings with QNaN values (see IEEE 754)
						Channels[iChannelIndex].SampleBuffer.PadQNaN(1L);
					}
				}
				
#if DEBUG
				printf("TC08Device::PollUnitThreaded: Read error, device state is now %i.\n",usbDevice->GetDeviceState());
#endif
				if(usbDevice->GetDeviceState()==PICODEVICE_STATE_DISCONNECTED) {
#if DEBUG
					printf("TC08Device::PollUnitThreaded: Exiting thread.\n");
#endif
					pthread_exit(NULL);
				} else {
					continue;
				}
			} else {
#if DEBUG
				printf("%i bytes ok!\n",readCount);
#endif
			}
			
			// Normal streaming mode constrains and scales readings
			//  and reports whether they've overflowed
			
			if (CJCUsed()) {
				if (!buf[0]) {
					flCJCTemp = std::numeric_limits<float>::quiet_NaN();
					
					// Store the CJC if it was enabled
					if (Channels[USBTC08_CHANNEL_CJC].fEnabled)
						Channels[USBTC08_CHANNEL_CJC].SampleBuffer.PadQNaN(1L);
				} else {
					// Convert CJC
					lADCCount = ConstructReading(buf[1], buf[2], buf[3]);
					Truncate20Bit(lADCCount);
					
					flMilliVolts = AdcToMilliVolts(USBTC08_GAIN_1V25, lADCCount);
					fConverted = TCTables.ConvertTemp(  USBTC08_TCTYPE_THERMISTOR,
														flMilliVolts,
														&flCJCTemp,
														NULL); // CJC never overflows
					assert(fConverted);
					
					// Store the CJC if it was enabled
					if (Channels[USBTC08_CHANNEL_CJC].fEnabled) {
						Channels[USBTC08_CHANNEL_CJC].SampleBuffer.AddReading(flCJCTemp);
					}
				}
			} else {
				flCJCTemp = std::numeric_limits<float>::quiet_NaN();
			}
			
			//
			// Now do the same for the thermocouple channels
			//
			for (iChannelIndex = 1; iChannelIndex <= USBTC08_MAX_CHANNELS; ++iChannelIndex) {
				fOverflow = false;
				
				if (Channels[iChannelIndex].fEnabled) {
					if (!buf[iChannelIndex*4]) {
						// Conversion error
						//
						Channels[iChannelIndex].SampleBuffer.PadQNaN(1L);
					} else if (isnan(flCJCTemp)&&(Channels[iChannelIndex].bTCType!='X')) {
						// Conversion requires CJC, which is unavailable
						//
						Channels[iChannelIndex].SampleBuffer.PadQNaN(1L);
					} else {
						// Convert thermocouple
						//
						lADCCount = ConstructReading(buf[(iChannelIndex*4)+1], buf[(iChannelIndex*4)+2], buf[(iChannelIndex*4)+3]);
						Truncate20Bit(lADCCount);
						flMilliVolts=AdcToMilliVolts(USBTC08_GAIN_THERMOCOUPLE, lADCCount);
						
						fOverflow|=!LimitReading(&flMilliVolts, THERMOCOUPLE_MV_LIMIT);
						
						if (Channels[iChannelIndex].bTCType == 'X') {
							flScaledValue = flMilliVolts;
						} else {
							fConverted = TCTables.ConvertMillivolts(Channels[iChannelIndex].bTCType,
																	flCJCTemp,
																	&flCJCMilliVolts,
																	&tempOverflow);
							assert(fConverted);
							fOverflow |= tempOverflow;
							
							flMilliVolts += flCJCMilliVolts;
							fConverted = TCTables.ConvertTemp(Channels[iChannelIndex].bTCType,
															  flMilliVolts,
															  &flScaledValue,
															  &tempOverflow);
							assert(fConverted);
							fOverflow |= tempOverflow;
						}
						
						Channels[iChannelIndex].SampleBuffer.AddReading(flScaledValue, fOverflow);
					}
				}
			}
			
		}
		
		
		do {					
#if DEBUG
			printf("TC08Device::PollUnitThreaded: Locking thread lock\n");
#endif
			pthread_mutex_lock(&threadLock);
			terminate = threadStop;
#if DEBUG
			printf("TC08Device::PollUnitThreaded: Unlocking thread lock\n");
#endif
			pthread_mutex_unlock(&threadLock); // Synchronised with USBTC08_Stop()
			if(terminate) {
#if DEBUG
			printf("TC08Device::PollUnitThreaded: Breaking while loop\n");
#endif
				
				break;
			}
			
#if DEBUG
			printf("TC08Device::PollUnitThreaded: Sleeping...\n");
#endif
			usleep(USBTC08_MIN_INTERVAL_MS*1000);
			
			#if DEBUG
				printf("TC08Device::PollUnitThreaded: GetTickCount()-LastSampleTime)<this->interval ((%i-%i==%i)<%i)\n",GetTickCount(),LastSampleTime,GetTickCount()-LastSampleTime,this->interval);
			#endif
		} while((GetTickCount()-LastSampleTime)<interval);
	} while(!terminate);
	
#if DEBUG
	//printf("TC08Device::PollUnitThreaded: Unlocking thread lock (after while)\n");
#endif
	
#if DEBUG
	printf("TC08Device::PollUnitThreaded: Exiting thread.\n");
#endif
	pthread_exit(NULL);
}

