/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright 2009--2026 by Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Qt includes
#include <QString>
#include <QIODevice>


/////////////////////// pappsomspp includes


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"
#include "MsXpS/libXpertMassCore/RecentFilesHandler.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{

RecentFilesHandler::RecentFilesHandler()
{
}

RecentFilesHandler::~RecentFilesHandler()
{
}

int
RecentFilesHandler::size() const
{
  return m_filePathsTimeUsed.size();
}

void
RecentFilesHandler::setMaxItemCount(int count)
{
  if(count < 0)
    return;
  m_maxItemCount = count;
}

int
RecentFilesHandler::getMaxItemCount() const
{
  return m_maxItemCount;
}

void
RecentFilesHandler::addItem(const QString &file_path)
{
  // The file_path is not the key to the QMap, it is the value, so
  // we may end up having the same file_path opened many different times.

  QPair<QDateTime, QString> item = getItem(file_path);

  if(item.first.isValid())
    removeItem(file_path);

  m_filePathsTimeUsed.insert(QDateTime::currentDateTime(), file_path);

  // QDateTime sorts in increasing dates. Older dates are at the beginning
  // because they are smaller.
  if(m_filePathsTimeUsed.size() > m_maxItemCount)
    m_filePathsTimeUsed.remove(m_filePathsTimeUsed.firstKey());
}

void
RecentFilesHandler::removeItem(const QString &file_path)
{
  QMap<QDateTime, QString>::iterator the_iterator = m_filePathsTimeUsed.begin();
  QMap<QDateTime, QString>::iterator the_iterator_end =
    m_filePathsTimeUsed.end();

  while(the_iterator != the_iterator_end)
    {
      if(the_iterator.value() == file_path)
        {
          m_filePathsTimeUsed.remove(the_iterator.key());
          return;
        }
      ++the_iterator;
    }
}

QPair<QDateTime, QString>
RecentFilesHandler::getItem(const QString &file_path)
{
  QMap<QDateTime, QString>::iterator the_iterator = m_filePathsTimeUsed.begin();
  QMap<QDateTime, QString>::iterator the_iterator_end =
    m_filePathsTimeUsed.end();

  while(the_iterator != the_iterator_end)
    {
      if(the_iterator.value() == file_path)
        {
          return QPair<QDateTime, QString>(the_iterator.key(),
                                           the_iterator.value());
        }
      ++the_iterator;
    }
  return QPair<QDateTime, QString>(QDateTime(), "");
}

QPair<QDateTime, QString>
RecentFilesHandler::getItem(QDateTime &date_time)
{
  QMap<QDateTime, QString>::iterator the_iterator =
    m_filePathsTimeUsed.find(date_time);
  if(the_iterator != m_filePathsTimeUsed.end())
    return QPair<QDateTime, QString>(the_iterator.key(), the_iterator.value());

  return QPair<QDateTime, QString>(QDateTime(), "");
}

QStringList
RecentFilesHandler::allItemsSortedByTime(bool most_recent_first)
{
  // Could be this, but we want to be able to order by time
  // return m_filePathsTimeUsed.values();

  QStringList file_paths;

  QMap<QDateTime, QString>::iterator the_iterator = m_filePathsTimeUsed.begin();
  QMap<QDateTime, QString>::iterator the_iterator_end =
    m_filePathsTimeUsed.end();

  while(the_iterator != the_iterator_end)
    {
      if(most_recent_first)
        file_paths.prepend(the_iterator.value());
      else
        file_paths.append(the_iterator.value());
      ++the_iterator;
    }

  return file_paths;
}

void
RecentFilesHandler::clear()
{
  m_filePathsTimeUsed.clear();
}

QByteArray
RecentFilesHandler::makeDataForSettings() const
{
  QByteArray data;

  QDataStream out_stream(&data, QIODevice::WriteOnly);
  out_stream.setVersion(QDataStream::Qt_6_0);
  out_stream << m_filePathsTimeUsed;


  return data;
}

void
RecentFilesHandler::makeFromSettingsData(const QByteArray &data)
{
  QByteArray local = data;

  QDataStream in_stream(&local, QIODevice::ReadOnly);
  in_stream.setVersion(QDataStream::Qt_6_0);
  in_stream >> m_filePathsTimeUsed;

  // qDebug() << "The paths:" << allItemsSortedByTime();
}

} // namespace libXpertMassCore
} // namespace MsXpS
