/* Copyright 2017 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

// Contains benchmark functions used with the code-generated benchmarks that can
// be used to test a model on android. See also code generation rules in
// tfcompile.bzl.
//
// This is separate from the built-in micro-benchmarks, because we want to:
// 1. show a binary with minimal dependencies, to show a close-to-lower-bound
//    binary size.
// 2. compile on Android.
#ifndef TENSORFLOW_COMPILER_AOT_BENCHMARK_H_
#define TENSORFLOW_COMPILER_AOT_BENCHMARK_H_

#include <functional>
#include <string>
#include <vector>

#include "tensorflow/core/platform/types.h"

namespace tensorflow {
namespace tfcompile {
namespace benchmark {

// Options specifies options for benchmarks of functions generated by tfcompile.
struct Options {
  // kDefaultMicros specifies the default time to run the benchmark, and is used
  // if neither max_iters nor max_micros is set.
  static const int64 kDefaultMicros = 3000000;

  int64 max_iters = 0;   // Maximum iterations to run, ignored if <= 0.
  int64 max_micros = 0;  // Maximum microseconds to run, ignored if <= 0.
};

// Stats holds statistics collected during benchmarking.
struct Stats {
  std::vector<int64> per_iter_us;  // Per-iteration deltas in us.
  int64 total_us;                  // Total time in us.

  Stats() : total_us(0) { per_iter_us.reserve(5000); }
};

// DumpStatsToStdout printfs to stdout stats in a multi-line human-friendly
// form.
void DumpStatsToStdout(const Stats& stats);

// BenchmarkFn is the signature of the function generated by tfcompile.
typedef std::function<void()> BenchmarkFn;

// Benchmark runs a benchmark of the function `fn`, collecting stats in `stats`.
// Use `options` to configure benchmarking options.
void Benchmark(const Options& options, const BenchmarkFn& fn, Stats* stats);

}  // namespace benchmark
}  // namespace tfcompile
}  // namespace tensorflow

#endif  // TENSORFLOW_COMPILER_AOT_BENCHMARK_H_
