/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform rational b-splines                           */
/*                                                                      */
/*            Copyright 2015, 2016 by Kay F. Jahnke                     */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/*! \file prefilter_poles.cc

    \brief calculates the poles of the b-spline prefilter using gsl and BLAS

    this doesn't have to be done for installing vspline if poles.cc is
    already present. Providing degrees up to 24 is just about what gsl
    can handle, with such high degrees the evaluation becomes quite imprecise
    as well, especialy for floats.

    compile with:
    clang++ -O3 -std=c++11 prefilter_poles.cc -oprefilter_poles -lgsl -lblas
    
    run
    
    ./prefilter_poles > poles.cc
    
    In this latest version, the double precision roots calculated by
    gsl/blas are now polished as long doubles to the best precision we can
    provide.

    TODO: could do with some TLC...
*/

#include <iostream>
#include <iomanip>

#include <vigra/array_vector.hxx>
#include <vigra/splines.hxx>
#include <gsl/gsl_poly.h>
#include <vspline/basis.h>

using namespace std ;
using namespace vigra ;

// newton() is an implementation of the newton method to approach a zero
// of a polynomial from a starting point nearby. We use this routine to
// polish the zeros we get from gsl/blas, since these values are only
// calculated in double precision and we want to take the precision as
// high as we possibly can.
// newton() calculates the augmented value from the current value, it's
// run in a loop until some termination criterion is satisfied.

template < typename dtype >
dtype newton ( dtype p , int ncoeff , dtype * coeff )
{
  dtype * pk = coeff + ncoeff - 1 ; // point to last coefficient
  dtype pot = 1 ;                   // powers of p
  int ex = 0 ;                      // exponent yielding the power
  int xd = 1 ;                      // ex derivative, d(x^n) = n(x^(n-1))
  dtype fx = 0 ;                    // f(p)
  dtype dfx = 0 ;                   // f'(p)
  
  // we work our way back to front, starting with p^0 and raising
  // the power with each iteration
  
  fx += pot * *pk ;
  for ( ; ; )
  {
    --pk ;
    if ( pk == coeff )
      break ;
    dfx += pot * xd * *pk ;
    xd++ ;
    ex++ ;
    pot = pow ( p , ex ) ;
    fx += pot * *pk ;
  }
  pot *= p ;
  fx += pot * *pk ;
//   std::cout << "fx: " << fx << " dfx: " << dfx << std::endl ;
  dtype next = p - fx / dfx ;
//   std::cout << "p: " << p << " p': " << next << std::endl ;
  return next ;
}

template < class T >
ArrayVector<long double> 
calculatePrefilterCoefficients(int DEGREE)
{
  ArrayVector<long double> res;
  const int r = DEGREE / 2;
  double a[2*r+1] ;
  long double la[2*r+1] ;
  double z[4*r+2] ;
  cout << "const long double K" << DEGREE << "[] = {" << endl ;
  // we calculate the basis function values at 0.5 intervals
  int imax = 2 * r ;
  if ( DEGREE & 1 )
    imax++ ;
  for(int i = 0; i <= imax ; ++i)
  {
    long double v
      = vspline::cdb_bspline_basis_2<long double>
        ( i , DEGREE , 0 ) ;

    cout << " " << v << "L ,   // basis(" << i * .5 << ")" << endl ;
    if ( ! ( i & 1 ) )
    {
      // for even i, we put the value in a[] as well - only even i
      // correspond to the value of the basis function at integral values
      // which we need for the poles
      int ih = i / 2 ;
      a [ r - ih ] = a [ r + ih ] = v ;
      la [ r - ih ] = la [ r + ih ] = v ;
    }
  }
// alternatively, gen_bspline_basis can be used to the same effect,
// the result is identical.
//
//   for(int i = 0; i <= imax ; ++i)
//   {
//     long double half_i = i / (long double) 2.0 ;
//     long double v
//       = vspline::gen_bspline_basis<long double>
//         ( half_i , DEGREE , 0 ) ;
// 
//     cout << " " << v << "L ,   // basis(" << half_i << ")" << endl ;
//     if ( ! ( i & 1 ) )
//     {
//       // for even i, we put the value in a[] as well - only even i
//       // correspond to the value of the basis function at integral values
//       // which we need for the poles
//       int ih = i / 2 ;
//       a [ r - ih ] = a [ r + ih ] = v ;
//       la [ r - ih ] = la [ r + ih ] = v ;
//     }
//   }
  cout << " } ; " << endl ;
      
  if(DEGREE > 1)
  {
    ArrayVector<double> roots;

    // we set up the environment gsl needs to find the roots
    gsl_poly_complex_workspace * w 
            = gsl_poly_complex_workspace_alloc (2*r+1);
    // now we call gsl's root finder
          gsl_poly_complex_solve (a, 2*r+1, w, z);
    // and release it's workspace
          gsl_poly_complex_workspace_free (w);

    // we only look at the real parts of the values, which are stored
    // interleaved real/imag. And we take them back to front, even though
    // it doesn't matter which end we start with - but conventionally
    // Pole[0] is the root with the largest absolute, so I stick with that.
          for(int i = 2 * r - 2 ; i >= 0; i-=2)
              if(VIGRA_CSTD::fabs(z[i]) < 1.0)
                  res.push_back(z[i]);
  }
//   // can use eigen alternatively:
//   {
//     using namespace Eigen ;
//     
//     Eigen::PolynomialSolver<long double, Eigen::Dynamic> solver;
//     Eigen::Matrix<long double,Dynamic, 1 >  coeff(2*r+1);
//     
//     long double * pa = a ;
//     for ( int i = 0 ; i < 2*r+1 ; i++ )
//     {
//       coeff[i] = *pa++ ;
//     }
//     
//     solver.compute(coeff);
//     
//     const Eigen::PolynomialSolver<long double, Eigen::Dynamic>::RootsType & r
//       = solver.roots();
//     
//         for ( int i = r.rows() - 1 ; i >= 0 ; i-- )
//     {
//       if ( std::fabs ( r[i].real() ) < 1 )
//         res.push_back(r[i].real() );
//     }
//   }
  // we polish the roots with the newton method
  for ( auto & p : res )
  {
    long double pa = p , pb = 0 ;
    while ( ( pa - pb ) * ( pa - pb ) >= LDBL_EPSILON )
    {
      pb = pa ;
      pa = newton ( pa , 2*r+1 , la ) ;
    }
//     std::cout << "....... settled on " << pa << std::endl ;
    p = pa ;
  }
  return res;
}

// TODO ugly mishmash of prints and calculations...

void print_poles ( int degree )
{
  ArrayVector<long double> res = calculatePrefilterCoefficients<double> ( degree ) ;
  if ( degree > 1 )
  {
    cout << "const long double Poles_" << degree << "[] = {" << endl ;
    for ( auto r : res )
      cout << r << "L ," << endl ;
    cout << "} ;" << endl ;
  }
}

int main ( int argc , char * argv[] )
{
  cout << setprecision(std::numeric_limits<long double>::max_digits10) ;
  
  for ( int degree = 0 ; degree < 25 ; degree++ )
    print_poles(degree) ;
  
  cout << noshowpos ;
  cout << "const long double* const precomputed_poles[] = {" << endl ;
  cout << "  0, " << endl ;
  cout << "  0, " << endl ;
  for ( int i = 2 ; i < 25 ; i++ )
    cout << "  Poles_" << i << ", " << endl ;
  cout << "} ;" << endl ;
  cout << "const long double* const precomputed_basis_function_values[] = {" << endl ;
  for ( int i = 0 ; i < 25 ; i++ )
    cout << "  K" << i << ", " << endl ;
  cout << "} ;" << endl ;
}
