/*
 * wreport/varinfo - Variable information
 *
 * Copyright (C) 2005--2011  ARPA-SIM <urpsim@smr.arpa.emr.it>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 * Author: Enrico Zini <enrico@enricozini.com>
 */

#ifndef WREPORT_VARINFO_H
#define WREPORT_VARINFO_H

#include <stdint.h>
#include <string>

namespace wreport {

/** @file
 * @ingroup core
 * Implement fast access to information about WMO variables.
 *
 * The measured value of a physical quantity has little meaning without
 * specifying what quantity it represents, what units are used to measure it,
 * and how many digits are significant for the value.
 *
 * This module provides access to all this metadata:
 *
 * \li \b wreport::Varcode represents what is the quantity measured, and takes
 *    values from the WMO B tables used for BUFR and CREX encodings.
 *    The ::WR_VAR macro can be used to construct wreport::Varcode values, and the
 *    ::WR_VAR_F, ::WR_VAR_X and ::WR_VAR_Y macros can be used to access the
 *    various parts of the dba_varcode.
 * \li \b wreport::Varinfo contains all the expanded information about a variable:
 *    its wreport::Varcode, description, measurement units, significant digits,
 *    minimum and maximum values it can have and other information useful for
 *    serialisation and deserialisation of values.
 *
 * There are many B tables with slight differences used by different
 * meteorological centre or equipment.  This module allows to access 
 * different vartables using dba_vartable_create().
 *
 * wreport::Vartable and wreport::Varinfo have special memory management: they are never
 * deallocated.  This is a precise design choice to speed up passing and
 * copying wreport::Varinfo values, that are used very intensely as they accompany
 * all the physical values processed by DB-All.e and its components.
 * This behaviour should not be a cause of memory leaks, since a software would
 * only need to access a limited amount of B tables during its lifetime.
 *
 * To construct a wreport::Varcode value one needs to provide three numbers: F, X
 * and Y.
 *
 * \li \b F (2 bits) identifies the type of table entry represented by the
 * dba_varcode, and is always 0 for B tables.  Different values are only used
 * during encoding and decoding of BUFR and CREX messages and are not in use in
 * other parts of DB-All.e.
 * \li \b X (6 bits) identifies a section of the table.
 * \li \b Y (8 bits) identifies the value within the section.  
 *
 * The normal text representation of a wreport::Varcode for a WMO B table uses the
 * format Bxxyyy.
 */

/**
 * Holds the WMO variable code of a variable
 */
typedef short unsigned int Varcode;

/// Format a varcode into a string
std::string varcode_format(Varcode code);

/**
 * Create a WMO variable code from its F, X and Y components.
 */
#define WR_VAR(f, x, y) ((wreport::Varcode)( ((unsigned)(f)<<14) | ((unsigned)(x)<<8) | (unsigned)(y) ))

/**
 * Convert a XXYYY string to a WMO variable code.
 *
 * This is useful only in rare cases, such as when parsing tables; use
 * descriptor_code() to parse proper entry names such as "B01003" or "D21301".
 */
#define WR_STRING_TO_VAR(str) ((wreport::Varcode)( \
		(( ((str)[0] - '0')*10 + ((str)[1] - '0') ) << 8) | \
		( ((str)[2] - '0')*100 + ((str)[3] - '0')*10 + ((str)[4] - '0') ) \
))

/**
 * Get the F part of a WMO variable code.
 */
#define WR_VAR_F(code) (((code) >> 14) & 0x3)
/**
 * Get the X part of a WMO variable code.
 */
#define WR_VAR_X(code) ((code) >> 8 & 0x3f)
/**
 * Get the Y part of a WMO variable code.
 */
#define WR_VAR_Y(code) ((code) & 0xff)

/**
 * Convert a FXXYYY string descriptor code into its short integer
 * representation.
 *
 * @param desc
 *   The 6-byte string descriptor as FXXYYY
 *
 * @return
 *   The short integer code that can be queried with the WR_GET_* macros
 */
Varcode descriptor_code(const char* desc);


/**
 * Describes how a wreport::Varinfo has been altered: it is used for supporting
 * variables coming from BUFR and CREX messages that use C codes to alter
 * variable information.
 */
typedef short unsigned int Alteration;

/**
 * Varinfo flags
 * @{
 */
#define VARINFO_FLAG_STRING 0x01  ///< Mark string variables
#define VARINFO_FLAG_BINARY 0x02  ///< Mark literal binary variables
/**@}*/


/**
 * Holds the information about a DBALLE variable.
 *
 * It never needs to be deallocated, as all the Varinfo returned by DB-ALLe
 * are pointers to memory-cached versions that are guaranteed to exist for all
 * the lifetime of the program.
 */
struct _Varinfo
{
	/** The variable code.  See @ref WR_VAR, WR_VAR_X, WR_VAR_Y. */
	Varcode var;
	/** The variable description. */
	char desc[64];
	/** The measurement unit of the variable. */
	char unit[24];
	/** The scale of the variable.  When the variable is represented as an
	 * integer, it is multiplied by 10**scale */
	int scale;
	/** The reference value for the variable.  When the variable is represented
	 * as an integer, and after scaling, it is added this value */
	int ref;
	/** The length in digits of the integer representation of this variable
	 * (after scaling and changing reference value) */
	unsigned len;
	/** The reference value for bit-encoding.  When the variable is encoded in
	 * a bit string, it is added this value */
	int bit_ref;
	/** The length in bits of the variable when encoded in a bit string (after
	 * scaling and changing reference value) */
	unsigned bit_len;
	/// Variable flags (see VARINFO_FLAG_* constants)
	unsigned flags;
	/** Minimum unscaled value the field can have */
	int imin;
	/** Maximum unscaled value the field can have */
	int imax;
	/** Minimum scaled value the field can have */
	double dmin;
	/** Maximum scaled value the field can have */
	double dmax;
	/** C-table alteration that has been applied to this entry (deprecated) */
	Alteration alteration;
	/** Other altered versions of this Varinfo */
	mutable struct _Varinfo* alterations;
	/** The measurement unit of the variable when encoded in BUFR. */
	char bufr_unit[24];
	/** The scale of the variable when encoded in BUFR. */
	int bufr_scale;

	/// Reference count
	mutable int _ref;

	_Varinfo();

    /// Increment the reference count to this Data object
    void do_ref() const { ++_ref; }

    /**
     * Decrement the reference count to this Data object, and return true
     * if the reference count went down to 0
     */
    bool do_unref() const { return (--_ref) == 0; }

	/// Check if we are a string value
	bool is_string() const
	{
		return (flags & VARINFO_FLAG_STRING) != 0;
	}

    /// Check if we are a binary value
    bool is_binary() const
    {
        return (flags & VARINFO_FLAG_BINARY) != 0;
    }

    /**
     * Encode a double value into an integer value using Varinfo decimal
     * encoding informations (ref and scale)
     *
     * @param fval
     *   Value to encode
     * @returns
     *   The double value encoded as an integer
     */
    int encode_int(double fval) const throw ();

    /**
     * Encode a double value into an integer value using Varinfo binary encoding
     * informations (bit_ref and bufr_scale)
     *
     * @param fval
     *   Value to encode
     * @returns
     *   The double value encoded as an integer
     */
    unsigned encode_bit_int(double fval) const;

	/**
	 * Decode a double value from integer value using Varinfo encoding
	 * informations
	 *
	 * @param val
	 *   Value to decode
	 * @returns
	 *   The decoded double value
	 */
	double decode_int(int val) const throw ();

	/**
	 * Decode a double value from integer value using Varinfo encoding
	 * informations for BUFR
	 *
	 * @param val
	 *   Value to decode
	 * @returns
	 *   The decoded double value
	 */
	double bufr_decode_int(uint32_t val) const throw ();

	/**
	 * Set all fields to 0, except the reference count
	 */
	void reset();

	/**
	 * Set the values all in one shot.
	 *
	 * It also calls compute_range
	 */
	void set(Varcode var, const char* desc, const char* unit, int scale = 0, int ref = 0, int len = 0, int bit_ref = 0, int bit_len = 0, int flags = 0, const char* bufr_unit = 0, int bufr_scale = 0);

	/**
	 * Initialise the varinfo to represent a string variable
	 *
	 * @param var the variable code
	 * @param desc the variable description
	 * @param len the maximum string length
	 */
	void set_string(Varcode var, const char* desc, int len);

	/**
	 * Compute the valid variable range and store it in the *min and *max
	 * fields
	 */
	void compute_range();
};

class Varinfo;

/// Smart pointer to handle/use varinfos
class MutableVarinfo
{
protected:
    /// Varinfo structure to which the pointer refers
    _Varinfo* m_impl;

public:
    //@{
    /// Create a smart pointer to the given variable information
    MutableVarinfo(_Varinfo* impl) : m_impl(impl) { m_impl->do_ref(); }
    MutableVarinfo(const MutableVarinfo& vi) : m_impl(vi.m_impl) { m_impl->do_ref(); }
    //@}
    ~MutableVarinfo() { if (m_impl->do_unref()) delete m_impl; }

    //@{
    /// Standard smart pointer methods
    MutableVarinfo& operator=(const MutableVarinfo& vi)
    {
        vi.m_impl->do_ref();
        if (m_impl->do_unref()) delete m_impl;
        m_impl = vi.m_impl;
        return *this;
    }
    _Varinfo* operator->() { return m_impl; }
    _Varinfo& operator*() { return *m_impl; }
    //@}

    /// Access the underlying _Varinfo structure
    _Varinfo* impl() const { return m_impl; }

	/**
	 * Create a single use varinfo structure.
	 *
	 * A single use varinfo structure is not memory managed by a vartable
	 * and needs to be deallocated explicitly when it is not needed
	 * anymore.
	 *
	 * The various fields of the resulting varinfo will be zeroed.
	 */
	static MutableVarinfo create_singleuse();

    friend class wreport::Varinfo;
};

/// Smart pointer to handle/use varinfos
class Varinfo
{
protected:
    /// Varinfo structure to which the pointer refers
    const _Varinfo* m_impl;

public:
    //@{
    /// Create a smart pointer to the given variable information
    Varinfo(const _Varinfo* impl) : m_impl(impl) { m_impl->do_ref(); }
    Varinfo(const _Varinfo& impl) : m_impl(&impl) { m_impl->do_ref(); }
    Varinfo(const Varinfo& vi) : m_impl(vi.m_impl) { m_impl->do_ref(); }
    Varinfo(const MutableVarinfo& vi) : m_impl(vi.m_impl) { m_impl->do_ref(); }
    //@}
    ~Varinfo() { if (m_impl->do_unref()) delete m_impl; }

    //@{
    /// Standard smart pointer methods
    const Varinfo& operator=(const Varinfo& vi)
    {
        vi.m_impl->do_ref();
        if (m_impl->do_unref()) delete m_impl;
        m_impl = vi.m_impl;
        return *this;
    }
    const _Varinfo& operator*() const { return *m_impl; }
    const _Varinfo* operator->() const { return m_impl; }
    //@}

    /// Access the underlying _Varinfo structure
    const _Varinfo* impl() const { return m_impl; }
};

}

#endif
/* vim:set ts=4 sw=4: */
